import ipaddress
from ipaddress import IPv4Address, IPv6Address, IPv4Network, IPv6Network
from lclazy import LazyLoader
import logging
from socket import AddressFamily, inet_ntoa
import struct
import toml
from typing import Optional, Union
from .typehints import ArpEntry, Lease
from .os_utils import (
    get_distro,
    get_arp_table,
)

psutil = LazyLoader('psutil', globals(), 'psutil')

logger = logging.getLogger(__name__)


# --------------------------------------------------------------------
class ValidIP:
    """
    Wrapper for ipaddress library to support IPv4 and IPv6
    as well as get True / False returns instead
    of AddressValueError.
    """

    # ----------------------------------------------------------------------
    def __init__(self, ip: str) -> None:
        self.ip_string = ip

        try:
            self.ip: Optional[Union[IPv4Address, IPv6Address]] = ipaddress.ip_address(ip)
        except ValueError:
            self.ip = None

    # ----------------------------------------------------------------------
    def valid_ip(self):
        return self.ip is not None

    # ----------------------------------------------------------------------
    def is_v4(self) -> bool:
        return isinstance(self.ip, IPv4Address)

    # ----------------------------------------------------------------------
    def is_v6(self) -> bool:
        return isinstance(self.ip, IPv6Address)

    # ----------------------------------------------------------------------
    def is_private(self) -> bool:
        return self.ip is not None and self.ip.is_private

    # ----------------------------------------------------------------------
    def is_network(self):
        try:
            return IPv4Network(self.ip_string)
        except (ipaddress.AddressValueError, ipaddress.NetmaskValueError, IndexError):
            try:
                return IPv6Network(self.ip_string)
            except (ipaddress.AddressValueError, ipaddress.NetmaskValueError, IndexError):
                return False


# --------------------------------------------------------------------
class NetworkInfo:

    def __init__(self):
        self._network_config = {}
        self._adapters = {}

    # ----------------------------------------------------------------
    def distro(self) -> str:
        return get_distro()

    # ----------------------------------------------------------------
    def adapters(self):
        if not self._adapters:
            self._adapters = psutil.net_if_addrs()
        return self._adapters

    # ----------------------------------------------------------------
    def default_gateway(self):
        """
        Get the system default gateway.
        """
        with open("/proc/net/route") as fh:
            for line in fh:
                fields = line.strip().split()
                if fields[1] != '00000000' or not int(fields[3], 16) & 2:
                    # If not default route or not RTF_GATEWAY, skip it
                    continue

                return inet_ntoa(struct.pack("<L", int(fields[2], 16)))

    # ----------------------------------------------------------------
    def is_gateway(self) -> bool:
        """
        System is network gateway
        """
        return self.get_config_param('MODE').lower() == 'gateway'

    # ----------------------------------------------------------------
    def arp_table(self) -> list[ArpEntry]:
        """
        Return arp table, scrubbed to include only local IPs,
        and excluding interface IPs.

        Some networks have switches where proxy-arp is enabled, so the IP
        is bound to the MAC of the switch, in which case the MAC will
        appear many times in the ARP cache. In those cases, the MAC is
        meaningless in determining device type.

        Include mac_frequency key to make it easier to detect proxy-arp addresses.
        """
        interface_ips = set(self.get_interface_ips())

        scrubbed = []
        for entry in get_arp_table(include_random=True):
            if entry['IP address'] in interface_ips:
                continue
            if not ValidIP(entry['IP address']).is_private():
                continue
            scrubbed.append(entry)

        return scrubbed

    # ----------------------------------------------------------------
    def dhcp_leases(self) -> list[Lease]:
        """
        Parse DHCP lease table, returning list of
        tuples of ip address, mac address and hostname
        """
        leases = []

        try:
            with open('/var/lib/dnsmasq/dnsmasq.leases', 'r') as dt:
                table = dt.readlines()
        except FileNotFoundError:
            return []

        for line in table:
            if not line.strip() or line.startswith('#'):
                continue

            try:
                _, mac, ip, host, _ = line.split()
                leases.append(Lease(ip, host, mac))
            except Exception:
                continue

        return leases

    # ----------------------------------------------------------------------
    def network_config(self) -> dict[str, str]:
        """
        Get default network settings file

        # Network mode
        MODE="gateway"

        # Network interface roles
        EXTIF="eth10"
        LANIF="br0"
        DMZIF=""
        HOTIF=""

        # Domain and Internet Hostname
        DEFAULT_DOMAIN="thinkwell.lan"
        INTERNET_HOSTNAME="thinkwelldesigns.com"

        # Extra LANS
        EXTRALANS=""
        """
        if not self._network_config:

            cf = '/etc/clearos/network.conf'
            try:
                self._network_config = toml.load(cf)
            except FileNotFoundError:
                logger.info(f'Config file does not exist: {cf}')

        return self._network_config

    # ----------------------------------------------------------------------
    def get_config_param(self, param: str) -> str:
        """
        Takes config lines from ClearOS config file
        Returns desired param value, stripping off quotes

        :param param: Desired param / key
        """
        return self.network_config().get(param) or ''

    # ----------------------------------------------------------------------
    def builtin_hostname(self) -> str:
        """
        Get default hostname of the system
        """
        return self.get_config_param(param='INTERNET_HOSTNAME')

    # ----------------------------------------------------------------------
    def lan_interface(self) -> str:
        """
        Returns LAN interface of system whether system is
        Gateway or Standalone.
        """
        try:
            return self.lan_interfaces()[0]
        except IndexError:
            return ''

    # ----------------------------------------------------------------------
    def lan_interfaces(self) -> list[str]:
        """
        Get all LAN hardware interfaces.
        """
        if self.is_gateway():
            interface = 'LANIF'
        else:
            interface = 'EXTIF'

        return self._valid_hardware_interfaces(interface)

    # ----------------------------------------------------------------------
    def vpn_interfaces(self) -> list[str]:
        """
        Get all OpenVPN / Wireguard / IPSec VPN interfaces.
        """
        return [
            itf for itf in self.adapters()
            if itf.startswith(('wg', 'tun', 'ppp')) and not itf.endswith('-ifb')
        ]

    # ----------------------------------------------------------------------
    def wan_interfaces(self) -> list[str]:
        """
        Get all WAN hardware interfaces.
        """
        return self._valid_hardware_interfaces('EXTIF')

    # ----------------------------------------------------------------------
    def dmz_interfaces(self) -> list[str]:
        """
        Get all DMZ hardware interfaces.
        """
        return self._valid_hardware_interfaces('DMZIF')

    # ----------------------------------------------------------------------
    def get_interface_ip(self, interface: str) -> str:
        """
        Get the IP address of the specified
        interface (eth1, enp0s20f0)

        :param interface: Interface type of system
        """
        try:
            adapter = self.adapters()[interface]
        except KeyError:
            logger.info('%s not a valid interface', interface)
            return ''

        for address in adapter:
            if address.family == AddressFamily.AF_INET:
                return address.address

        logger.info(f'Unable to calculate IP address for {interface}')
        return ''

    # ----------------------------------------------------------------------
    def get_interface_ips(self) -> list[str]:
        """
        Get list of all valid IPs on the system
        """
        ips = set()

        for interface, addresses in self.adapters().items():
            if interface == 'lo':
                continue
            for addr in addresses:
                if addr.family == AddressFamily.AF_INET or addr.family == AddressFamily.AF_INET6:
                    ips.add(addr.address)

        return list(ips)

    def valid_interfaces(self, param) -> set[str]:
        """
        Get all the valid interfaces for a given type:
        LANIF / EXTIF / DMZIF.

        ClearOS doesn't necessarily prune old interfaces from config files. :weary:
        Only return interfaces from config files if they are present on the system.
        """
        interfaces = set(self.get_config_param(param).split())
        existing_interfaces = set(self.adapters().keys())
        return existing_interfaces.intersection(interfaces)

    def _valid_hardware_interfaces(self, param) -> list[str]:
        """
        Get all the valid hardware interfaces for a given type:
        LANIF / EXTIF / DMZIF.
        """
        interfaces = self.valid_interfaces(param)
        return [intf for intf in interfaces if ':' not in intf and '.' not in intf]

    def _valid_virtual_interfaces(self, param) -> list[str]:
        """
        Get all the valid virtual interfaces for a given type:
        LANIF / EXTIF / DMZIF.
        """
        interfaces = self.valid_interfaces(param)
        return [intf for intf in interfaces if ':' in intf or '.' in intf]


__all__ = (
    'NetworkInfo',
    'ValidIP',
)
