# Use modified `run` workflow so that output is returned.
# Callers need to analyze output to decide if processes should be restarted.
import os
import sys
import subprocess
import pprint as pprint_lib
from .exceptions import RsyncError
from .command_maker import get_rsync_command
from .typehints import RsyncInputs, RsyncResult, RsyncResultException

pprint = pprint_lib.PrettyPrinter(indent=1).pprint


def run(cwd=None, strict=True, verbose=False, timeout=60 * 3, **kwargs) -> RsyncResult:
    rsync_command = get_rsync_command(**kwargs)
    rsync_string = ' '.join(rsync_command)

    if cwd is None:
        cwd = os.getcwd()

    if verbose is True:
        print(f'[lcrsync runner] running command (timeout={timeout} secs):\n{rsync_command}')

    subprocess_result = run_command(rsync_command, cwd=cwd, timeout=timeout)

    if strict is True:
        code = subprocess_result["return_code"]
        _check_return_code(code, rsync_string, subprocess_result)

    return subprocess_result


def run_command(commands: list, timeout: int = 3, cwd=None) -> RsyncResult:
    """
    wrapper on subprocess.run()
    returns results in dict along with many diagnostics for failure cases
    Taken from https://github.com/pjcrosbie/sysrsync
    """

    commands = [str(elem) for elem in commands]

    try:
        completed_process = subprocess.run(
            commands,
            check=True,  # exception on non-zero error code
            timeout=timeout,  # exception on timeout secs
            capture_output=True,
            universal_newlines=True,
            cwd=cwd,
        )
    except subprocess.CalledProcessError as error:
        result: RsyncResult = {
            "result": "CalledProcessError",
            "cmd": error.cmd,
            "return_code": error.returncode,
            "stdout": error.stdout,
            "stderr": error.stderr,
        }
    except subprocess.TimeoutExpired as error:
        result: RsyncResult = {
            "result": "TimeoutExpired",
            "cmd": error.cmd,
            "stdout": error.stdout,
            "stderr": error.stderr,
            "return_code": 1,
        }
    except Exception:
        err_type, err_value, traceback = sys.exc_info()
        result: RsyncResultException = {
            "result": "UnexpectedError",
            "type": err_type,
            "value": err_value,
            "traceback": traceback,
            "cmd": '',
            "stdout": '',
            "stderr": '',
            "return_code": 1,
        }
        # HACK: for testing, uncomment this to get crash here
        # raise
    else:
        # all okay
        result: RsyncResult = {
            "result": "OK",
            "cmd": completed_process.args,
            "stdout": completed_process.stdout,
            "stderr": completed_process.stderr,
            "return_code": completed_process.returncode,
        }

    # these function inputs
    # note,
    #   cmd is a return item from subprocess.run()
    #   cmd_list is what we pass to subprocess.run()
    #   cmd_line should be cut and paste-able but
    #      not always, watch quoting args with spaces
    command_line = ' '.join([str(elem) for elem in commands])
    inputs: RsyncInputs = {
        "cmd_list": commands,
        "cmd_line": command_line,
        "cwd": cwd,
        "timeout": timeout,
    }
    result["inputs"] = inputs

    return result


def _check_return_code(return_code: int, action: str, subprocess_result: dict):
    if return_code != 0:
        pprint(subprocess_result)
        msg = f"[lcrsync runner] {action} exited with code {return_code}"
        raise RsyncError(msg)
