from datetime import datetime, timedelta, timezone
import json
from lclazy import LazyLoader

try:
    from lchttp.json import json_dumps
except (ImportError, ModuleNotFoundError):
    from json import dumps as json_dumps
import time
import logging
import secrets
from typing import Literal, TYPE_CHECKING
from .typehints import CID_STR, PasetoFooter, PasetoPayload, Subject

if TYPE_CHECKING:
    from aspen_crypto.keys import FacadePrivateKey, FacadePublicKey
    import jwt
    import pyseto
else:
    jwt = LazyLoader('jwt', globals(), 'jwt')
    pyseto = LazyLoader('pyseto', globals(), 'pyseto')

logger = logging.getLogger(__name__)
TIME_LEEWAY = 20


class Token:
    """
    Token class for generating signed JWTs. Pass to Request
    class so that each request gets its own signed token.
    """

    def __init__(
        self,
        private_key: 'FacadePrivateKey',
        auth_method: Literal["Signed_JWT", "Pinned_JWT"] = 'Signed_JWT',
        cid: str = '',
        email: str = '',
        public_key_cid: str = '',
        sub: str = '',
        sub_cid: str = '',
    ):
        self.private_key = private_key
        self.auth_method = auth_method
        self.cid = cid
        self.email = email
        self.public_key_cid = public_key_cid
        self.public_key = private_key.public_key
        self.private_key_pem = private_key.as_pem
        self.sub = sub
        self.sub_cid = sub_cid

    def __repr__(self) -> str:
        name = self.__class__.__name__
        return (
            f"{name}('<private_key>', {self.auth_method!r}, "
            f"{self.cid!r}, {self.email!r}, {self.public_key_cid!r})"
        )

    def __str__(self) -> str:
        return f"{self.auth_method} <token>"

    def auth_header(self) -> str:
        """
        Create an HTTP Authorization header

        Authorization: JWT 401f7ac837da42b97f613d789819ff93537bee6a
        """
        return f"{self.auth_method} {self.sign()}"

    def sign(self) -> str:
        """
        Create and return signed authentication JWT
        """
        token_data = {
            "time": int(time.time()),
            "nonce": secrets.token_urlsafe(nbytes=8),
        }
        if self.sub:
            token_data['sub'] = self.sub
        if self.sub_cid:
            token_data['sub_cid'] = self.sub_cid
        if self.public_key_cid:
            token_data['public_key_cid'] = self.public_key_cid
        if self.cid:
            token_data['cid'] = str(self.cid)
        elif self.email:
            token_data['email'] = self.email

        headers = {
            "kid": self.private_key.public_key.fingerprint,
        }
        token = jwt.encode(
            payload=token_data,
            key=self.private_key_pem,
            algorithm=self.public_key.allowed_algorithms[0],
            headers=headers,
        )
        return token


class RegistrationToken(Token):
    """
    Use this token on the client side when registering
    new accounts / public keys on the server.
    """

    def __init__(self, *args, **kwargs):
        if 'auth_method' not in kwargs:
            kwargs['auth_method'] = 'Pinned_JWT'
        super().__init__(*args, **kwargs)


# ---------------------------------------------------------------------------
class PasetoToken:
    """
    Represents a Paseto Token that's either been constructed
    by our code or has been verified to be valid.
    """

    auth_method = 'Paseto'
    version: int = 4
    subject: Subject = Subject.Any
    user_cid: CID_STR = ''

    def __init__(
        self,
        private_key: 'FacadePrivateKey',
        user_cid: CID_STR = '',
        email: str = '',
        subject: Subject | None = None,
        subject_cid: CID_STR = '',
        verification_cid: CID_STR = '',
        expires: int = 90,
    ):
        self.private_key: 'FacadePrivateKey' = private_key
        self.public_key: 'FacadePublicKey' = private_key.public_key
        self.user_cid = user_cid or self.user_cid
        self.email: str = email
        self.subject: Subject = subject or self.subject
        self.subject_cid: CID_STR = subject_cid
        self.verification_cid: CID_STR = verification_cid
        self.expires = expires

    def __repr__(self) -> str:
        name = self.__class__.__name__
        return (
            f"{name}('<private_key>', {self.auth_method!r}, "
            f"{self.user_cid!r}, {self.email!r}, {self.expires!r})"
        )

    def __str__(self) -> str:
        return f"{self.auth_method} <token>"

    def auth_header(self) -> str:
        """
        Create an HTTP Authorization header.
        """
        return f"{self.auth_method} {self.sign().decode()}"

    def payload(self) -> PasetoPayload:
        """
        Return data for Paseto token payload.
        """
        now = datetime.now(tz=timezone.utc)
        payload = PasetoPayload(
            nonce=secrets.token_urlsafe(nbytes=8),
            exp=(now + timedelta(seconds=self.expires)).isoformat(timespec="seconds"),
            nbf=(now - timedelta(seconds=TIME_LEEWAY)).isoformat(timespec="seconds"),
        )

        if self.user_cid:
            payload['user_cid'] = self.user_cid
        elif self.email:
            payload['email'] = self.email

        if self.subject:
            payload['sub'] = self.subject
        if self.subject_cid:
            payload['sub_cid'] = self.subject_cid

        return payload

    def footer(self) -> PasetoFooter:
        """
        Insert keys into the footer to enable server-side
        verification, such as lookup of Public Keys to verify
        token signing.
        """
        if self.verification_cid:
            return PasetoFooter(cid=self.verification_cid)

        return PasetoFooter(kid=self.private_key.public_key.to_paserk_id(self.version))

    def sign(self) -> bytes:
        """
        Create and return signed authentication Paseto token
        """
        paseto = pyseto.Paseto.new(
            exp=self.expires,
            include_iat=True,
            leeway=TIME_LEEWAY,
        )

        return paseto.encode(
            key=self.private_key.as_paseto(self.version),
            payload=json_dumps(self.payload()),
            footer=json_dumps(self.footer()),
            serializer=json,
        )


# ---------------------------------------------------------------------------
class V1PasetoToken(PasetoToken):
    version = 1


class V2PasetoToken(PasetoToken):
    version = 2


class V3PasetoToken(PasetoToken):
    version = 3


V4PasetoToken = PasetoToken


__all__ = (
    'Token',
    'RegistrationToken',
    'PasetoToken',
    'V1PasetoToken',
    'V2PasetoToken',
    'V3PasetoToken',
    'V4PasetoToken',
)
