# Extracted from https://github.com/tensorflow/tensorflow/blob/master/tensorflow/python/util/lazy_loader.py
import importlib
import types

_LAZY_LOADER_PREFIX = "_ll"


class LazyLoader(types.ModuleType):
    """Lazily import a module, mainly to avoid pulling in large dependencies.

    `contrib`, and `ffmpeg` are examples of modules that are large and not always
    needed, and this allows them to only be loaded when they are used.
    """

    # The lint error here is incorrect.
    def __init__(self, local_name, parent_module_globals, name):
        self._ll_local_name = local_name
        self._ll_parent_module_globals = parent_module_globals

        # These members allow doctest correctly process this module member without
        # triggering self._load(). self._load() mutates parent_module_globals and
        # triggers a dict mutated during iteration error from doctest.py.
        # - for from_module()
        super().__setattr__("__module__", name.rsplit(".", 1)[0])
        # - for is_routine()
        super().__setattr__("__wrapped__", None)

        super().__init__(name)

    def _load(self):
        """Load the module and insert it into the parent's globals."""
        # Import the target module and insert it into the parent's namespace
        module = importlib.import_module(self.__name__)
        self._ll_parent_module_globals[self._ll_local_name] = module

        # Update this object's dict so that if someone keeps a reference to the
        #     LazyLoader, lookups are efficient (__getattr__ is only called on lookups
        #     that fail).
        self.__dict__.update(module.__dict__)

        return module

    def __getattr__(self, name):
        module = self._load()
        return getattr(module, name)

    def __setattr__(self, name, value):
        if name.startswith(_LAZY_LOADER_PREFIX):
            super().__setattr__(name, value)
        else:
            module = self._load()
            setattr(module, name, value)
            self.__dict__[name] = value
            try:
                # check if the module has __all__
                if name not in self.__all__ and name != "__all__":
                    self.__all__.append(name)
            except AttributeError:
                pass

    def __delattr__(self, name):
        if name.startswith(_LAZY_LOADER_PREFIX):
            super().__delattr__(name)
        else:
            module = self._load()
            delattr(module, name)
            self.__dict__.pop(name)
            try:
                # check if the module has __all__
                if name in self.__all__:
                    self.__all__.remove(name)
            except AttributeError:
                pass

    def __repr__(self):
        # Carefully to not trigger _load, since repr may be called in very
        # sensitive places.
        return f"<LazyLoader {self.__name__} as {self._ll_local_name}>"

    def __dir__(self):
        module = self._load()
        return dir(module)

    def __reduce__(self):
        return importlib.import_module, (self.__name__,)


__all__ = [
    'LazyLoader',
]
