from .common import clean_byte_string, json_default_vals

import orjson
from orjson import (
    OPT_NON_STR_KEYS,
    OPT_INDENT_2,
    OPT_PASSTHROUGH_DATETIME,
    OPT_SORT_KEYS,
    JSONDecodeError,
)


def orjson_dumps(
    data,
    indent=None,
    default=None,
    sort_keys=False,
    rtype='bytes',
) -> bytes | str:
    """
    orjson wrapper function for dumping python values to JSON,
    to make it easy to swap out JSON library project-wide.
    """

    # use OPT_PASSTHROUGH_DATETIME for isoformat, which doesn't round microseconds
    options = OPT_NON_STR_KEYS | OPT_PASSTHROUGH_DATETIME
    if indent:
        options |= OPT_INDENT_2
    if sort_keys:
        options |= OPT_SORT_KEYS

    try:
        value = orjson.dumps(
            data,
            default=default or json_default_vals,
            option=options,
        )
        return value if rtype == 'bytes' else value.decode('utf8')
    except TypeError:
        pass

    # orjson can't handle ints such as 654338975898042097013509947142486150416577742806
    # raise TypeError: Integer exceeds 64-bit range
    import json

    st_value = json.dumps(data, default=str)
    return st_value.encode('utf8') if rtype == 'bytes' else st_value


def orjson_loads(value: bytes | str):
    """
    orjson wrapper function for loading string to python values,
    for consistent value parsing and to make it easy to
    swap out JSON library project-wide.
    """
    try:
        return orjson.loads(value)
    except JSONDecodeError as e:
        try:
            return orjson.loads(clean_byte_string(value))
        except Exception:
            raise e


__all__ = (
    'orjson_dumps',
    'orjson_loads',
)
