from functools import lru_cache
from urllib.parse import urlparse


@lru_cache()
def extract_domain_name(url: str) -> str:
    """
    Return parsed URL object, normalizing URLs
    that can be served with http / https
    """
    if not url:
        return ''

    schemaless = url.startswith('//')
    if url.startswith(('#', '/')) and not schemaless:
        return ''

    if schemaless:
        url = f'https:{url}'

    pu = urlparse(url)
    netloc = pu.netloc or ''
    has_port = netloc.find(':')

    if has_port > -1:
        netloc = netloc[:has_port]

    if not netloc:
        return ''

    return public_suffix().get_public_suffix(netloc)


@lru_cache(maxsize=None)
def public_suffix():
    # expensive to import and initialize, so lru_cache
    from publicsuffix2 import PublicSuffixList
    return PublicSuffixList()


__all__ = (
    'extract_domain_name',
    'public_suffix',
)
