import yaml
from yaml.error import MarkedYAMLError

try:
    from yaml import CSafeLoader as SafeLoader
except ImportError:
    from yaml import SafeLoader  # type: ignore[assignment]
try:
    from yaml import CSafeDumper as SafeDumper
except ImportError:
    from yaml import SafeDumper  # type: ignore[assignment]

from .settings import CONF_DIR, ENCRYPT_KEYS


# ----------------------------------------------------------------------------
class YamlParser:
    """
    Yaml config file parser with similar API to LCConfigParser
    """

    conf_file: str = ''

    def __init__(self) -> None:
        self._values: dict = {}

    def load(self) -> dict:
        if not (values := self._values):
            try:
                with open(self.conf_file, 'r') as cf:
                    try:
                        values = yaml.load(cf, SafeLoader)
                    except MarkedYAMLError:
                        print(f'unable to load {self.conf_file}')
                        return {}
            except FileNotFoundError:
                return {}

        # Config loader doesn't support yaml files
        # that load as anything other than dict
        if not isinstance(values, dict):
            values = {}

        if encrypt_key_values := ENCRYPT_KEYS.intersection(values):
            from aspen_crypto.encryption import decrypt
            from cryptography.fernet import InvalidToken

            for key in encrypt_key_values:
                try:
                    values[key] = decrypt(values[key])
                except InvalidToken:
                    continue

        self._values = values

        return self._values or {}

    def as_typed_dict(self):
        return self.load()

    def save(self, data):
        if encrypt_key_values := ENCRYPT_KEYS.intersection(data):
            from aspen_crypto.encryption import encrypt

            for key in encrypt_key_values:
                data[key] = encrypt(str(data[key]))

        with open(self.conf_file, 'w') as cf:
            yaml.dump(
                data=data,
                stream=cf,
                Dumper=SafeDumper,
                encoding='utf-8',
                allow_unicode=True,
                default_flow_style=False,
            )
        self._values = data or {}

    def __str__(self):
        return f'{self.__class__.__name__} config file at {self.conf_file}'


# ----------------------------------------------------------------------------
class AuthActiveDirectoryYmlConfig(YamlParser):
    conf_file = f'{CONF_DIR}/auth_active_directory.conf'


# ----------------------------------------------------------------------------
class AuthLdapYmlConfig(YamlParser):
    conf_file = f'{CONF_DIR}/auth_ldap.conf'


# ----------------------------------------------------------------------------
class ConcordiaYmlConfig(YamlParser):
    conf_file = f'{CONF_DIR}/concordia.yml'


__all__ = (
    'AuthActiveDirectoryYmlConfig',
    'AuthLdapYmlConfig',
    'ConcordiaYmlConfig',
)
