from ast import literal_eval
from configparser import ConfigParser, SectionProxy
from os.path import expanduser
from typing import Any, Self
from .settings import CONF_DIR, ENCRYPT_KEYS


# ----------------------------------------------------------------------------
class LCConfigParser(ConfigParser):
    main_section = ''
    config_settings = ''
    persist_file = ''

    def __init__(self, *args, **kwargs):
        self.main_section = kwargs.pop('main_section', '') or self.main_section
        self.config_settings = kwargs.pop('config_settings', '') or self.config_settings
        self.persist_file = kwargs.pop('persist_file', '') or self.persist_file

        super().__init__(*args, **kwargs)
        self._values = {}
        self._loaded = False

    def load(self) -> Self:
        """
        Load the ConfigParser object by calling the correct "read" method
        based on the value of `config_settings`.

        `config_settings` can be a string file path, dict, or StringIO
        """
        if self._loaded:
            return self

        if isinstance(self.config_settings, str):
            self.read(get_path(self.config_settings))

        elif isinstance(self.config_settings, dict):
            self.read_dict(self.config_settings)

        else:
            # Enable easy testing by using StringIO object
            from io import StringIO

            if isinstance(self.config_settings, StringIO):
                self.read_file(self.config_settings)

        self._loaded = True

        return self

    def reload(self) -> None:
        """
        Reset cached properties and reload from disk.

        Called when saving config data to disk, to ensure that
        latest version of the file has been read so that saving
        changes doesn't stomp on other sections.
        """
        self._values = {}
        self._loaded = False
        self.load()

    def getfloat(self, *args: Any, **kwargs: Any) -> float:
        """
        Override getfloat to make sure that int values
        are not processed with this converter
        """
        section, option, *_ = args
        value = str(self[section][option])

        try:
            int(value)
            is_int = True
        except ValueError:
            is_int = False

        if is_int:
            raise ValueError(f'"{value}" is an integer. Use "getint" converter.')

        return super().getfloat(*args, **kwargs)

    def getdict(self, *args: Any, **kwargs: Any) -> dict:  # noqa
        section, option, *_ = args
        value = self[section][option]

        if isinstance(value, str) and value.startswith('{'):
            try:
                return literal_eval(value)
            except Exception:  # noqa
                pass

        if isinstance(value, dict):
            return value

        raise ValueError(f'"{value}" is not a dictionary.')

    def getlist(self, *args: Any, **kwargs: Any) -> list:  # noqa
        section, option, *_ = args
        value = self[section][option]

        if isinstance(value, str) and value.startswith('['):
            try:
                return literal_eval(value)
            except Exception:  # noqa
                pass

        if isinstance(value, list):
            return value

        raise ValueError(f'"{value}" is not a list.')

    def getnone(self, *args: Any, **kwargs: Any) -> None | str:  # noqa
        section, option, *_ = args
        if (value := self[section][option]) == 'None':
            return None
        return value

    def as_typed_dict(self, section: str = '') -> Any:
        """
        Return section as dictionary with values
        cast to python types
        """
        section = section or self.main_section
        self.load()

        cached_values = self._values.get(section, {})
        if cached_values or not self.has_section(section):
            return cached_values

        values = {}
        converters = [f'get{conv}' for conv in self.converters.keys()]

        for k in self[section]:
            for converter in converters:
                try:
                    values[k] = getattr(self[section], converter)(k)
                    break
                except ValueError:
                    continue
            else:
                values[k] = self[section][k]

        if encrypt_key_values := ENCRYPT_KEYS.intersection(values):
            from aspen_crypto.encryption import decrypt
            from cryptography.fernet import InvalidToken

            for key in encrypt_key_values:
                try:
                    values[key] = decrypt(values[key])
                except InvalidToken:
                    continue

        self._values[section] = values

        return values

    def save(self) -> None:
        """
        Save entire config file to disk
        """
        if not self.persist_file:
            raise ValueError('Specify a persist_file location')

        with open(get_path(self.persist_file), 'w') as cf:
            self.write(cf)

    def save_section(self, section: str = '', data: dict | None = None) -> None:
        """
        Save data to the specified section
        """
        if not data:
            return

        section = section or self.main_section
        self.reload()
        if not self.has_section(section):
            self.add_section(section)

        if encrypt_key_values := ENCRYPT_KEYS.intersection(data):
            from aspen_crypto.encryption import encrypt

            for key in encrypt_key_values:
                data[key] = encrypt(str(data[key]))

        for k, v in data.items():
            self[section][k] = str(v)

        self.save()


# ----------------------------------------------------------------------------
class APIServerConfig(LCConfigParser):
    """
    API / Sync testing works best on live servers.
    Test code will load server info from this file.
    """

    config_settings = f'{CONF_DIR}/apiservers.conf'
    persist_file = f'{CONF_DIR}/apiservers.conf'


# ----------------------------------------------------------------------------
class AspenCtlSettingsConfig(LCConfigParser):
    main_section = 'Account'
    config_settings = '~/.aspenctl.conf'
    persist_file = '~/.aspenctl.conf'


# ----------------------------------------------------------------------------
class CareCenterSettingsConfig(LCConfigParser):
    config_settings = f'{CONF_DIR}/CareCenter.conf'
    persist_file = f'{CONF_DIR}/CareCenter.conf'


# ----------------------------------------------------------------------------
class ClavisAuthSettingsConfig(LCConfigParser):
    main_section = 'Compass'
    config_settings = f'{CONF_DIR}/clavis.conf'
    persist_file = f'{CONF_DIR}/clavis.conf'


# ----------------------------------------------------------------------------
class ConsoleSettingsConfig(LCConfigParser):
    config_settings = f'{CONF_DIR}/settings.conf'
    persist_file = f'{CONF_DIR}/settings.conf'

    # -------------------------------------------------------------------------
    def rebranded_system(self) -> bool:
        """
        Check if this Console is rebranded to reseller's theme
        """
        self.load()
        try:
            return self['BoxData'].getboolean('rebranded') or False
        except KeyError:
            return False

    # -------------------------------------------------------------------------
    def email_section(self) -> str:
        if self.rebranded_system():
            return 'rebranded_email'
        return 'default_email'

    # -------------------------------------------------------------------------
    def email_settings(self) -> SectionProxy:
        """
        Get correct email settings, based on
        whether system is rebranded or not
        """
        section = self.email_section()
        if not self.has_section(section):
            self.add_section(section)
        return self[section]  # noqa

    def save_email(self, data: dict) -> None:
        """
        Save Email settings to correct section

        :param data: Dictionary of settings to save
        """
        from aspen_crypto.encryption import encrypt

        section = self.email_settings()

        for k, v in data.items():
            if k == 'password':
                # Password field isn't required on the form,
                # so don't override with null value
                if v:
                    section[k] = encrypt(v)
                continue

            section[k] = str(v)

        self.save()


# ----------------------------------------------------------------------------
class ConsoleImpersonatorConfig(ConsoleSettingsConfig):
    """
    Use this class when a system is the impersonator of another system,
    in a backup capacity.

    This class accesses the original settings of the local system.
    """

    config_settings = f'{CONF_DIR}/impersonator.conf'
    persist_file = f'{CONF_DIR}/impersonator.conf'


# ----------------------------------------------------------------------------
class DynamicDNSConfig(LCConfigParser):
    main_section = 'DDNSURL'
    config_settings = f'{CONF_DIR}/dynamic_dns.conf'
    persist_file = f'{CONF_DIR}/dynamic_dns.conf'


# ----------------------------------------------------------------------------
class CapsuleConfig(LCConfigParser):
    main_section = 'Compass'
    config_settings = f'{CONF_DIR}/capsule.conf'
    persist_file = f'{CONF_DIR}/capsule.conf'


# ----------------------------------------------------------------------------
class ObjectStorageConfig(LCConfigParser):
    main_section = 'Compass'
    config_settings = f'{CONF_DIR}/object_storage.conf'
    persist_file = f'{CONF_DIR}/object_storage.conf'


# ----------------------------------------------------------------------------
class MinioConfig(LCConfigParser):
    main_section = 'Compass'
    config_settings = f'{CONF_DIR}/minio.conf'
    persist_file = f'{CONF_DIR}/minio.conf'


# ----------------------------------------------------------------------------
class PeopleConfig(LCConfigParser):
    config_settings = f'{CONF_DIR}/people.conf'
    persist_file = f'{CONF_DIR}/people.conf'


# ----------------------------------------------------------------------------
class SymmetricKeysConfig(LCConfigParser):
    main_section = 'SymmetricKeys'
    config_settings = f'{CONF_DIR}/.symmetric_keys.conf'
    persist_file = f'{CONF_DIR}/.symmetric_keys.conf'


# ----------------------------------------------------------------------------
class SyncPublisherConfig(LCConfigParser):
    main_section = 'SyncPublisher'
    config_settings = f'{CONF_DIR}/syncpublisher.conf'
    persist_file = f'{CONF_DIR}/syncpublisher.conf'


# ----------------------------------------------------------------------------
class LocalLDAPServer:
    """
    Parse nslcd.conf file and load into auth_ldap.conf
    file which is readable by the logcabin username.
    """

    config_settings = '/etc/nslcd.conf'

    def __init__(self):
        self._base: str = ''
        self._domain: str = ''
        self._tld: str = ''
        self._uri: str = ''
        self._ldap_version: str = ''
        self._config_lines: list = []
        self._binddn: str = ''
        self._bindpw: str | None = None
        self._search_scope: str = ''
        self._ssl: str = ''
        self._tls_cert: str = ''
        self._tls_key: str = ''

    def load(self):
        """
        Load configuration file into lines. If file does not exist,
        fail silently.
        """
        if not self._config_lines:
            try:
                with open(get_path(self.config_settings), 'r') as ldap_cfg:
                    config_lines = [cl.strip() for cl in ldap_cfg.readlines()]
                    self._config_lines = [cl for cl in config_lines if cl]
            except FileNotFoundError:
                pass

        return self._config_lines

    @property
    def base(self) -> str:
        if not self._base:
            self._base = self.get_key('base')
        return self._base

    @property
    def domain(self) -> str:
        """
        Get domain name from search base:
          * dc=ksl,dc=lan
          * dc=bv,dc=mss-eggs,dc=com
        """
        if self.base and not self._domain:
            split_base = self.base.split(',')
            try:
                self._domain = split_base[-2][3:]
            except Exception:  # noqa
                pass
        return self._domain

    @property
    def tld(self) -> str:
        """
        Get top level domain from
        dc=ksl,dc=lan.
        """
        if self.base and not self._tld:
            split_base = self.base.split(',')
            try:
                self._tld = split_base[-1][3:]
            except Exception:  # noqa
                pass
        return self._tld

    @property
    def uri(self) -> str:
        if not self._uri:
            uri = self.get_key('uri') or ''
            self._uri = uri.strip('/')
        return self._uri

    @property
    def binddn(self) -> str:
        if not self._binddn:
            self._binddn = self.get_key('binddn')
        return self._binddn

    @property
    def bindpw(self) -> str:
        if not self._bindpw:
            self._bindpw = self.get_key('bindpw')
        return self._bindpw

    @property
    def ssl(self) -> str:
        if not self._ssl:
            self._ssl = self.get_key('ssl')
        return self._ssl

    @property
    def tls(self) -> str:
        return self.ssl

    @property
    def search_scope(self) -> str:
        if not self._search_scope:
            self._search_scope = self.get_key_by_comment(key='scope', comment='search scope')
        return self._search_scope

    def get_key(self, key: str) -> str:
        """
        Return value of key, separated by spaces.

        uid nslcd
        gid ldap
        ldap_version 3
        """
        for line in self._config_lines:
            if line.startswith(key):
                first_space = line.index(' ')
                value = line[first_space:].strip()
                return value

        return ''

    def get_key_by_comment(self, key: str, comment: str) -> str:
        """
        Some keys, such as `scope` can occur multiple times in the config file.
        Will have preceding comment indicating the value type. The next "key"
        line after that comment is the one we want.

        # The default search scope.
        scope sub
        #scope one
        #scope base

        # The distinguished name of the search base.
        base dc=ksl,dc=lan
        """
        found_preceding_comment = False

        if not self._search_scope:
            for line in self._config_lines:
                if not found_preceding_comment:
                    if not line.startswith('#'):
                        continue
                    elif comment in line:
                        found_preceding_comment = True
                        continue

                if not line.startswith(key):
                    continue

                first_space = line.index(' ')
                value = line[first_space:].strip()
                return value

        return ''

    def __getattr__(self, item):
        try:
            super().__getattr__(item)  # type: ignore[misc]
        except AttributeError as e:
            ldap_attr = self.get_key(item)
            if ldap_attr:
                return ldap_attr
            raise e


def get_path(path: str) -> str:
    """
    Get full path, expanding user path if necessary.

    >>> get_path('~/.config.ini')
    /home/dave/.config.ini

    >>> get_path('/etc/aspen/.config.ini')
    /etc/aspen/.config.ini
    """
    if not path.startswith('~'):
        return path

    home = expanduser('~')

    return f'{home}{path[1:]}'


__all__ = (
    'APIServerConfig',
    'AspenCtlSettingsConfig',
    'LCConfigParser',
    'LocalLDAPServer',
    'CapsuleConfig',
    'ObjectStorageConfig',
    'MinioConfig',
    'CareCenterSettingsConfig',
    'ClavisAuthSettingsConfig',
    'ConsoleSettingsConfig',
    'ConsoleImpersonatorConfig',
    'DynamicDNSConfig',
    'PeopleConfig',
    'SymmetricKeysConfig',
    'SyncPublisherConfig',
)
