import datetime
import pytest
import pytz
from celery import schedules
from django.contrib.contenttypes.models import ContentType
from django.core.exceptions import ValidationError
from django.test import TestCase, override_settings
from django.utils import timezone
from django.conf import settings

import timezone_field

from lc_celery_beat.models import (
    crontab_schedule_celery_timezone,
    ClockedSchedule,
    IntervalSchedule,
    SolarSchedule,
    PeriodicTask,
    PeriodicTasks,
    DAYS,
)
from t.proj.models import O2OToPeriodicTasks


class CrontabScheduleTestCase(TestCase):
    FIRST_VALID_TIMEZONE = timezone_field.\
        TimeZoneField.default_choices[0][0].zone

    def test_default_timezone_without_settings_config(self):
        assert crontab_schedule_celery_timezone() == "UTC"

    @override_settings(CELERY_TIMEZONE=FIRST_VALID_TIMEZONE)
    def test_default_timezone_with_settings_config(self):
        assert crontab_schedule_celery_timezone() == self.FIRST_VALID_TIMEZONE


class SolarScheduleTestCase(TestCase):
    EVENT_CHOICES = SolarSchedule._meta.get_field("event").choices

    def test_celery_solar_schedules_sorted(self):
        assert all(
            self.EVENT_CHOICES[i] <= self.EVENT_CHOICES[i + 1]
            for i in range(len(self.EVENT_CHOICES) - 1)
        ), "SolarSchedule event choices are unsorted"

    def test_celery_solar_schedules_included_as_event_choices(self):
        """Make sure that all Celery solar schedules are included
        in SolarSchedule `event` field choices, keeping synchronized
        Celery solar events with `lc_celery_beat` supported solar
        events.

        This test is necessary because Celery solar schedules are
        hardcoded at models so that Django can discover their translations.
        """
        event_choices_values = [value for value, tr in self.EVENT_CHOICES]
        for solar_event in schedules.solar._all_events:
            assert solar_event in event_choices_values

        for event_choice in event_choices_values:
            assert event_choice in schedules.solar._all_events


class IntervalScheduleTestCase(TestCase):

    def test_duplicate_schedules(self):
        kwargs = {'every': 1, 'period': IntervalSchedule.SECONDS}
        schedule, created = IntervalSchedule.objects.get_or_create(kwargs)

        self.assertTrue(created)
        self.assertIsInstance(schedule, IntervalSchedule)

        new_schedule, created = IntervalSchedule.objects.get_or_create(kwargs)
        self.assertFalse(created)
        self.assertIsInstance(new_schedule, IntervalSchedule)


class ClockedScheduleTestCase(TestCase):

    def test_duplicate_schedules(self):
        kwargs = {'clocked_time': timezone.now()}
        schedule, created = ClockedSchedule.objects.get_or_create(kwargs)

        self.assertTrue(created)
        self.assertIsInstance(schedule, ClockedSchedule)

        new_schedule, created = ClockedSchedule.objects.get_or_create(kwargs)
        self.assertFalse(created)
        self.assertIsInstance(new_schedule, ClockedSchedule)

    # IMPORTANT: we must have a valid time-zone (not UTC) to do an accurate testing
    @override_settings(TIME_ZONE='Africa/Cairo')
    def test_timezone_format(self):
        """Make sure the scheduled time is not shown in UTC when time zone is used"""
        tz_info = pytz.timezone(settings.TIME_ZONE).localize(datetime.datetime.utcnow())
        schedule, created = ClockedSchedule.objects.get_or_create(clocked_time=tz_info)
        # testing str(schedule) calls make_aware() internally
        self.assertEqual(str(schedule.clocked_time), str(schedule))


@pytest.mark.django_db()
class PeriodicTaskTestCase(TestCase):

    def test_lookup_schedule(self):
        kwargs = {'every': 1, 'period': IntervalSchedule.SECONDS}
        schedule, created = IntervalSchedule.objects.get_or_create(kwargs)
        self.assertTrue(created)

        task, created = PeriodicTask.objects.get_or_create(
            name='This is a hard job',
            task='console.tasks.hard_job',
            args=['hard', 'job', 1],
            schedule=schedule,
        )
        self.assertTrue(created)
        self.assertIsInstance(task, PeriodicTask)
        self.assertEqual(schedule, task.schedule_object)

        task, _ = PeriodicTask.objects.update_or_create(
            name='This is another hard job',
            task='console.tasks.more_jobs',
            defaults=dict(
                schedule=schedule,
                args=['hard', 'job', 2],
            ),
        )
        self.assertIsInstance(task, PeriodicTask)
        self.assertEqual(schedule, task.schedule_object)


@pytest.mark.django_db()
class OneToOneRelTestCase(TestCase):
    """
    Make sure that when OneToOne relation Model changed,
    the `PeriodicTasks.last_update` will be updated.
    """

    @classmethod
    def setUpTestData(cls):
        super().setUpTestData()
        cls.interval_schedule = IntervalSchedule.objects.create(every=10, period=DAYS)

    def test_trigger_update_when_saved(self):
        o2o_to_periodic_tasks = O2OToPeriodicTasks.objects.create(
            name='name1',
            task='task1',
            is_active=True,
            schedule_content_type=ContentType.objects.get_for_model(self.interval_schedule),
            schedule_object_id=self.interval_schedule.pk,
        )
        not_changed_dt = PeriodicTasks.last_change()
        o2o_to_periodic_tasks.enabled = True  # Change something on instance.
        o2o_to_periodic_tasks.save()
        has_changed_dt = PeriodicTasks.last_change()
        self.assertTrue(
            not_changed_dt != has_changed_dt, 'The `PeriodicTasks.last_update` has not be update.'
        )
        # Check the `PeriodicTasks` does be updated.

    def test_trigger_update_when_deleted(self):
        o2o_to_periodic_tasks = O2OToPeriodicTasks.objects.create(
            name='name1',
            task='task1',
            is_active=True,
            schedule_content_type=ContentType.objects.get_for_model(self.interval_schedule),
            schedule_object_id=self.interval_schedule.pk,
        )
        not_changed_dt = PeriodicTasks.last_change()
        o2o_to_periodic_tasks.delete()
        has_changed_dt = PeriodicTasks.last_change()
        self.assertTrue(
            not_changed_dt != has_changed_dt, 'The `PeriodicTasks.last_update` has not be update.'
        )
        # Check the `PeriodicTasks` does be updated.
