from datetime import datetime
from typing import Optional, Union

from django.utils import timezone

from lc_celery_beat.models import ClockedSchedule, PeriodicTask


def delete_future_task(name: str, task: str):
    """
    Delete a Period Task before it's called.
    Also deletes Schedule if no other tasks remain linked to it.
    """
    for task in PeriodicTask.objects.filter(name=name, task=task):
        schedule = task.schedule_object
        task.delete()
        if not schedule.builtin and not schedule.has_tasks():
            schedule.delete()


def schedule_future_task(
    date: datetime,
    name: str,
    task: str,
    queue: Optional[str] = None,
    task_args: Optional[Union[list, tuple]] = None,
    task_kwargs: Optional[dict] = None,
    check_existing: bool = False,
) -> PeriodicTask:
    """
    Schedule a task to be run once, at a precise time in the future.
    """
    if check_existing:
        current_task = task_already_scheduled(task)
        if current_task:
            return current_task

    schedule, _ = ClockedSchedule.objects.get_or_create(clocked_time=date)
    name = f'{name} - {date}'

    periodic_task, _ = PeriodicTask.objects.update_or_create(
        name=name,
        defaults=dict(
            schedule=schedule,
            name=name,
            task=task,
            args=task_args or [],
            kwargs=task_kwargs or {},
            queue=queue,
            one_off=True,
        )
    )
    if periodic_task and periodic_task.schedule_object_id != schedule.pk:
        periodic_task.schedule_object_id = schedule.pk
        periodic_task.save()

    return periodic_task


def task_already_scheduled(task) -> Optional[PeriodicTask]:
    """
    Check if task is already scheduled to run at some future time.
    """
    return PeriodicTask.objects.filter(
        task=task,
        clocked__clocked_time__gt=timezone.now(),
    ).first()


__all__ = (
    'schedule_future_task',
    'delete_future_task',
)
