#!/usr/bin/python3
import io
import re
import os
from os.path import dirname, join

from setuptools import setup


def read(*names, **kwargs):
    return io.open(join(dirname(__file__), *names), encoding=kwargs.get('encoding', 'utf8')).read()


def get_version():
    with open('ivylink/__init__.py', 'r') as f:
        line = f.readline()
        match = re.match(r'__version__ = \'([\d\.]+)\'', line)

        if not match:
            raise ImportError("Can't read the version of ivylink")

        version = match.group(1)
        return version


def get_packages(package):
    """
    Return root package and all sub-packages.
    """
    return [
        dirpath for dirpath, dirnames, filenames in os.walk(package)
        if os.path.exists(os.path.join(dirpath, '__init__.py'))
    ]


setup(
    name='ivylink',
    version=get_version(),
    license='CF',
    description='Provide inspection tools for DNS names and Images',
    author='Dave Burkholder',
    author_email='dave@compassfoundation.io',
    url='https://code.compassfoundation.io/dave/ivylink',
    packages=get_packages('ivylink'),
    data_files=[
        ('/etc/supervisord.d', [os.path.join('config/supervisord', 'ivylink.conf')]),
        ('/etc/rsyslog.d', [os.path.join('config/rsyslog', 'ivylink.conf')]),
        ('/etc/logrotate.d', [os.path.join('config/logrotate', 'ivylink')]),
    ],
    zip_safe=True,
    classifiers=[
        'Intended Audience :: Developers',
        'Operating System :: Unix',
        'Operating System :: POSIX',
        'Programming Language :: Python :: 3.9',
        'Programming Language :: Python :: Implementation :: CPython',
        'Programming Language :: Python :: Implementation :: PyPy',
        'Topic :: Utilities',
    ],
    keywords=[
        'ivylink',
        'security',
    ],
    install_requires=[
        'annoy',
        'lchttp',
        'lcrequests',
        'numpy',
        'pdqhash',
        'pillow-simd',
        'pybase64',
        'sanic',
        'validators',
    ],
    options={
        'bdist_rpm': {
            'post_install': 'post_install.sh',
            'pre_install': 'pre_install.sh',
            'post_uninstall': 'post_uninstall.sh',
        },
    },
    entry_points={
        'console_scripts': [
            'ivy = ivylink.main_tasks:program.run',
            'ivylink = ivylink.main_tasks:program.run',
            'ivylink_web = ivylink.web.server:serve',
        ],
    },
)
