from io import BytesIO
from PIL import ImageFilter
import pybase64
from sanic import json
from sanic.log import logger
from sanic.exceptions import InvalidUsage
from ivylink.images import PDQHash, BlockImageIndex
from ..core.views import IvyLinkImageView

INAPPROPRIATE_IMAGE_INDEX = BlockImageIndex()


class BlurredImage(IvyLinkImageView):
    """
    POST base64-encoded image to this view to get
    a blurred image in response. Used to reduce
    possible exposure to objectionable content.
    """

    async def post(self, request):
        data = request.json

        image = self.load_image(data)
        try:
            image = image.filter(ImageFilter.GaussianBlur(self.get_blur_rate(image.size)))
        except Exception as e:
            raise InvalidUsage(str(e))

        img_file = BytesIO()
        image.save(img_file, format=self.get_image_format(data))

        return json({'image': pybase64.b64encode(img_file.getvalue()).decode()})

    def get_blur_rate(self, image_size: tuple) -> int:
        """
        Blur larger images increasingly more than small ones.
        """
        longest_dimension = max(image_size)
        if longest_dimension < 150:
            return 7
        if longest_dimension < 350:
            return 10
        if longest_dimension < 600:
            return 15
        if longest_dimension < 900:
            return 20
        if 1250 < longest_dimension < 2000:
            return 25
        return 30

    def get_image_format(self, data):
        """
        Get image format from the content type provided from Redwood.
        """
        content_type = (data.get('Content-Type') or 'image/jpeg').lower()

        if content_type.endswith(('jpeg', 'jpg')):
            return 'JPEG'

        if content_type.endswith('png'):
            return 'PNG'

        if content_type.endswith('webp'):
            return 'WebP'

        if content_type.endswith('ico'):
            return 'ICO'

        if content_type.endswith('gif'):
            return 'GIF'

        if content_type.endswith('tiff'):
            return 'TIFF'

        if content_type.endswith(('jp2', 'jpm', 'jpx')):
            return 'JPEG 2000'

        return 'JPEG'


class FlaggedImage(IvyLinkImageView):
    """
    POST base64-encoded image to this view to check if
    the image hash has been flagged as objectionable.
    """

    async def post(self, request):
        category = ''
        image = self.load_image(request.json)

        vectors = PDQHash(image).compute_dihedral()
        inappropriate = INAPPROPRIATE_IMAGE_INDEX.search(vectors)
        if inappropriate:
            logger.info('Classified %s as inappropriate %s' % (request.json['url'], inappropriate))
            category = 'inappropriate'

        return json({'category': category})


__all__ = (
    'BlurredImage',
    'FlaggedImage',
)
