from invoke import task
from lcrequests import Request
from ivylink.images import (
    HashDatabase,
    ImageIndex,
    AllowHashDatabase,
    AllowImageIndex,
    BlockHashDatabase,
    BlockImageIndex,
    PDQHash,
)
from ivylink.parsing import IvyHTMLParser, validate_selector


@task(
    help={'image': 'File path or URL to Image to be Allowed'},
)
def allow(ctx, image):  # noqa
    """
    Generate Allow Image Hash of the image at the provided path or URL.
    """
    hash_database = AllowHashDatabase()
    _hash_image(image, hash_database, AllowImageIndex(hash_database))


@task(
    help={'image': 'File path or URL to Image to be Blocked'},
)
def block(ctx, image):  # noqa
    """
    Generate Block Image Hash of the image at the provided path or URL.
    """
    hash_database = BlockHashDatabase()
    _hash_image(image, hash_database, BlockImageIndex(hash_database))


@task(
    help={
        'url': 'URL of page from which to extract images',
        'selector': 'Parent CSS selector of image(s) to extract from page',
    },
)
def page(ctx, url, selector):  # noqa
    """
    Specify a CSS selector containing img tags, and hash to block all child img tags found.
    """
    validate_selector(selector)
    req = Request(url)

    try:
        resp = req.get()
    except Exception as e:
        raise SystemExit(f'{e}')

    if resp.status_code != 200:
        raise SystemExit(f'Unable to retrieve {url} - Status: {resp.status_code}')

    images = IvyHTMLParser(resp.text).images_from_css(selector)
    hash_database = BlockHashDatabase()
    image_index = BlockImageIndex(hash_database)

    for image in images:
        url = image.attributes['src']
        if url.startswith('//'):
            url = f'https:{url}'
        try:
            _hash_image(url, hash_database, image_index, persist_changes=False)
        except SystemExit:
            print(f'Image hash already present in database - {url}')

    hash_database.save()


def _hash_image(
    image: str,
    hash_database: HashDatabase,
    hash_index: ImageIndex,
    persist_changes: bool = True,
):
    """
    Generate image hash to allow / block the specified image.
    """
    img_hash = PDQHash(image)
    dihedral_hashes = img_hash.compute_dihedral()
    isometric_hashes = img_hash.compute_isometric()

    if hash_index.search(dihedral_hashes):
        raise SystemExit(f'Image hash already present in database - {isometric_hashes.r0}')

    hash_database.add(image, isometric_hashes.r0)

    if persist_changes:
        hash_database.save()
        hash_index.refresh_from_database()

    action = 'block' if isinstance(hash_database, BlockHashDatabase) else 'allow'
    print(f'Added image hash {isometric_hashes.r0} from {image} to {action.title()} database file')


__all__ = (
    'allow',
    'block',
    'page',
)
