import numpy as np
import pdqhash
from typing import (
    List,
    Optional,
    Tuple,
    Union,
)
from ivylink.typehints import DihedralHash, IsometricHash
from .tools import (
    ImageInputType,
    string_to_vector,
    to_image_array,
    vector_to_string,
)


class PDQHash:
    """
    Inspired by https://github.com/thorn-oss/perception

    The Facebook PDQ hash. Based on the original implementation located at
    the `official repository <https://github.com/facebook/ThreatExchange>`.
    """

    distance_metric: str = "hamming"
    dtype: str = "bool"
    hash_length: int = 256
    hash_format: str = "base64"

    def __init__(self, image: ImageInputType):
        """
        image: An image represented as a filepath, PIL image object,
            or np.ndarray object - np.ndarray objects must be in RGB color order.
        """
        self.image = to_image_array(image)
        self._dihedral_hash: Optional[DihedralHash] = None
        self._computed_hash = None

    def string_to_vector(self, hash_string: str):
        """Convert hash string to vector.
        Args:
            hash_string: The input hash string
        """
        return string_to_vector(
            hash_string,
            dtype=self.dtype,
            hash_length=self.hash_length,
            hash_format=self.hash_format,
        )

    def vector_to_string(self, vector: np.ndarray) -> str:
        """Convert vector to hash string.
        Args:
            vector: Input vector
        """
        return vector_to_string(vector, dtype=self.dtype, hash_format=self.hash_format)

    def _compute(self):
        if not self._computed_hash:
            self._computed_hash = pdqhash.compute(self.image)
        return self._computed_hash

    def _compute_with_quality(self):
        hash_vector, quality = self._compute()
        return hash_vector > 0, quality

    def compute_with_quality(
        self
    ) -> Tuple[Union[np.ndarray, Optional[str], List[Optional[str]]], int]:
        """Compute hash and hash quality from image.
        Returns:
            A tuple of (hash, quality)
        """
        vector, quality = self._compute_with_quality()
        if self.hash_format == "vector":
            return vector, quality
        return self.vector_to_string(vector), quality

    def compute_dihedral(self) -> DihedralHash:
        """
        Compute hash of image returning vector hashes of 8 rotations.
        """
        if not self._dihedral_hash:
            hash_vectors, _ = pdqhash.compute_dihedral(self.image)
            self._dihedral_hash = DihedralHash(*hash_vectors)
        return self._dihedral_hash

    def compute_isometric(self) -> IsometricHash:
        """
        Compute hash of image returning string hashes of 8 rotations.
        """
        return IsometricHash(*[self.vector_to_string(vector) for vector in self.compute_dihedral()])
