from __future__ import print_function
import requests


class IRedMailApiBase(object):

    api_path = ''

    def __init__(self, url, session=None):
        self.session = session
        self.url = url
        self.username = None
        self.password = None


    def login(self, username=None, password=None):
        """
        Login to API, store cookies for future requests
        """

        if username:
            self.username = username

        if password:
            self.password = password

        json_obj = {'username': self.username, 'password': self.password}

        r = requests.post(
            self.build_url(custom_path='login'),
            data=json_obj
        )

        self.session = r.cookies


    def get_session(self):
        if not self.session:
            self.login()

        return self.session


    def build_url(self, api_name='', path_ext='', custom_path=''):
        """
        Build API URL with optional path extension.

        :param api_name: Default API path.  (i.e. email address, domain name, etc)
        Sample default return: (<domain>/api/user/<mail>

        :param path_ext: URL path extension for API calls that
        require it.
        Sample return: (<domain>/api/user/<mail>/keep_mailbox_days/90)

        :param custom_path: URL path overwrite for API calls that
        do not follow the conventional scheme.
        Sample return: (<domain>/api/verify_password/user/<mail>)
        """

        if path_ext:
            return '{url}/api/{api_path}/{path_ext}'.format(
                url=self.url,
                api_path=self.api_path,
                path_ext=path_ext
            )

        if custom_path:
            return '{domain}/api/{new_path}'.format(domain=self.url, new_path=custom_path)

        return '{domain}/api/{api_path}/{api_name}'.format(
            domain=self.url,
            api_path=self.api_path,
            api_name=api_name
        )


    def post(self, url, post_params):
        """
        Post data to API

        :param url: Complete URL
        :param post_params: JSON object to post
        """

        return requests.post(
            url,
            cookies=self.get_session(),
            data=post_params
        )


    def get(self, url):
        """
        Get data from API

        :param url: Complete URL
        """

        return requests.get(
            url,
            cookies=self.get_session()
        )


    def delete(self, url):
        """
        Delete data from API

        :param url: Complete URL
        """

        return requests.delete(
            url,
            cookies=self.get_session()
        )


    def put(self, url, put_params):
        """
        Put data to API

        :param url: Complete URL
        :param put_params: JSON object to put
        """

        return requests.put(
            url,
            cookies=self.get_session(),
            data=put_params
        )


    def exists(self, api_name):
        """
        Check if an object exists on API

        :param api_name: Default API path.  (i.e. email address, domain name, etc)
        :rtype: bool
        """

        url = self.build_url(api_name=api_name)

        resp = self.get(url)
        json_data = resp.json()

        if not json_data['_success']:
            return False

        return True


    def create(self, api_name, **params):
        """
        Post helper function

        :param api_name: Default API path.  (i.e. email address, domain name, etc)
        :param params: API params
        :return: Dict containing JSON response
        :rtype: dict
        """

        url = self.build_url(api_name=api_name)

        return self.post(url, params)


    def update(self, api_name, **params):
        """
        Put helper function

        :param api_name: Default API path.  (i.e. email address, domain name, etc)
        :param params: API params
        :return: Dict containing JSON response
        :rtype: dict
        """

        url = self.build_url(api_name=api_name)

        return self.put(url, params)


    def retrieve(self, api_name):
        """
        Get helper function

        :param api_name: Default API path.  (i.e. email address, domain name, etc)
        :return: Dict containing JSON response
        :rtype: dict
        """

        url = self.build_url(api_name=api_name)

        return self.get(url)


    def remove(self, api_name, days=None):
        """
        Delete helper function.  Days to keep deleted mail can
        be specified by using the days param.

        :param api_name: Default API path.  (i.e. email address, domain name, etc)
        :param days: Integer specifying the days to keep deleted mail
        :return: Dict containing JSON response
        :rtype: dict
        """

        if days:
            path = '{}/keep_mailbox_days/{}'.format(api_name, days)
            url = self.build_url(path_ext=path)

            return self.delete(url)

        url = self.build_url(api_name=api_name)

        return self.delete(url)


class Domain(IRedMailApiBase):

    api_path = 'domain'


class MailUser(IRedMailApiBase):

    api_path = 'user'


__all__ = [IRedMailApiBase, Domain, MailUser]


if __name__ == '__main__':
    import config

    domain = Domain(config.domain)
    mail_user = MailUser(config.domain)
    domain.login(username=config.api_username, password=config.api_password)
