#!/usr/bin/python3
import distro
import io
import re
import os
from os.path import dirname, join

from setuptools import setup


def read(*names, **kwargs):
    return io.open(join(dirname(__file__), *names), encoding=kwargs.get('encoding', 'utf8')).read()


def get_data_files():
    df = [
        ('/etc/compass', [os.path.join('config/etc', 'hedge.conf')]),
        ('/etc/rsyslog.d', [os.path.join('config/rsyslog', 'hedge.conf')]),
        ('/etc/logrotate.d', [os.path.join('config/logrotate', 'hedge')]),
    ]
    if distro.id() == 'clearos':
        init_script = ('/etc/supervisord.d', [os.path.join('config/supervisord', 'hedge.conf')])
    else:
        init_script = ('/lib/systemd/system', [os.path.join('config/lib', 'hedge.service')])

    df.append(init_script)

    return df


def get_version():
    with open('hedge/__init__.py', 'r') as f:
        line = f.readline()
        match = re.match(r'__version__ = \'([\d\.]+)\'', line)

        if not match:
            raise ImportError("Can't read the version of hedge")

        version = match.group(1)
        return version


def get_packages(package):
    """
    Return root package and all sub-packages.
    """
    return [
        dirpath for dirpath, dirnames, filenames in os.walk(package)
        if os.path.exists(os.path.join(dirpath, '__init__.py'))
    ]


setup(
    name='hedge',
    version=get_version(),
    license='CF',
    description='Monitor files on the system and act on file system events',
    author='Dave Burkholder',
    author_email='dave@compassfoundation.io',
    url='https://code.compassfoundation.io/dave/hedge',
    packages=get_packages('hedge'),
    package_data={
        '': ['*.html'],
    },
    data_files=get_data_files(),
    zip_safe=True,
    classifiers=[
        'Intended Audience :: Developers',
        'License :: SAP',
        'Operating System :: Unix',
        'Operating System :: POSIX',
        'Programming Language :: Python :: 3.11',
        'Programming Language :: Python :: 3.12',
        'Programming Language :: Python :: Implementation :: CPython',
        'Programming Language :: Python :: Implementation :: PyPy',
        'Topic :: Utilities',
    ],
    keywords=[
        'hedge',
        'security',
    ],
    install_requires=[
        'distro',
        'jinja2',
        'lcconfig',
        'py-zabbix',
        'tabulate',
        'syslogmp',
        'watchdog',
    ],
    options={
        'bdist_rpm': {
            'post_install': 'post_install.sh',
            'pre_install': 'pre_install.sh',
        },
    },
    entry_points={"console_scripts": [
        "hedge = hedge.__main__:main",
    ]},
)
