import collections
import datetime
import re

import syslogmp


class RSyslogMessage:
    def __init__(self, message):
        self._bytes: bytes = message
        self._parsed_logline: collections.namedtuple = None
        self._parsed_message: tuple = ()

    @property
    def parsed_logline(self) -> collections.namedtuple:
        if not self._parsed_logline:
            self._parsed_logline = syslogmp.parse(self._bytes)
        return self._parsed_logline

    @property
    def timestamp(self) -> datetime.datetime:
        return self.parsed_logline.timestamp

    @property
    def hostname(self) -> str:
        return self.parsed_logline.hostname

    @property
    def message(self) -> str:
        return self.parsed_logline.message.decode()

    @property
    def is_interesting(self) -> bool:
        """
        Define in subclass whether the
        message is meaningful or not
        """
        raise NotImplementedError


class SSHLogin(RSyslogMessage):

    @property
    def is_interesting(self) -> bool:
        """
        Return True if the message is a successful or failed
        ssh login from sshd. Return False if anything else
        """
        msg = self.message.lower()
        i = msg.index(':')
        return 'sshd' in msg and any([
            'accepted publickey' in msg,
            'failed publickey' in msg,
            'accepted password' in msg,
            'failed password' in msg,
            'invalid user' in msg[i+2:i+14],
        ])

    @property
    def parsed_message(self) -> tuple:
        """
        Parsing these scenarios
         * "sshd[8838]: Accepted publickey for root from 173.161.228.229 port 52119"
         * "sshd[2353]: Failed publickey for root from 104.218.187.15 port 40310"
         * "sshd[8838]: Invalid user rotten from 173.161.228.229 port 52114"
         * "sshd[8838]: Accepted password for root from 173.161.228.229 port 46454 ssh2"
         * "sshd[8838]: Failed password for root from 173.161.228.229 port 46450 ssh2"
         * "sshd[8838]: Failed password for invalid user blah from 173.161.228.229 port 46452 ssh2"
        """
        if not self._parsed_message:
            self._parsed_message = re.findall(
                r"(?:(Accepted|Failed) (publickey|password) for|Invalid user) (.+?) from ([\d\.]+?) port",
                self.message,
            )[0]
        return self._parsed_message

    @property
    def username(self) -> str:
        return self.parsed_message[2].replace('invalid user ', '')

    @property
    def ip(self) -> str:
        return self.parsed_message[3]

    @property
    def success(self) -> bool:
        verdict = self.parsed_message[0].lower()
        return 'accepted' in verdict

    @property
    def authtype(self) -> str:
        auth = self.parsed_message[1].lower()
        if not auth:  # Invalid user authorizing with ssh key
            return 'key'
        return 'publickey' in auth and 'key' or 'password'


__all__ = [
    'SSHLogin',
]
