"""
hedge Test Cases. Run entire suite by:

python3 -m unittest

Run individual test class by:

python3 -m unittest hedge.tests.test_<name>
"""
from pathlib import Path
import unittest
from ..actors.sshlogin import SSHLoginDatabase, SSHLogin
from ..installation import ssh_login_setup
from ..settings import DATABASE_PATH


interesting = [
    b"<86>Jun 18 15:40:01 system sshd[8838]: Accepted publickey for root from 173.161.228.229 port 52119",
    b"<86>Jun 18 15:40:02 system sshd[2353]: Failed publickey for root from 104.218.187.15 port 40310",
    b"<86>Jun 18 15:40:03 system sshd[8838]: Invalid user rotten from 173.161.228.229 port 52114",
    b"<86>Jun 18 15:40:04 system sshd[8838]: Accepted password for root from 173.161.228.229 port 46454 ssh2",
    b"<86>Jun 18 15:40:05 system sshd[8838]: Failed password for root from 173.161.228.229 port 46450 ssh2",
    b"<86>Jun 18 15:40:06 system sshd[8838]: Failed password for invalid user blah from 173.161.228.229 port 46452 ssh2",
]
uninteresting = [
    b"<86>Jun 18 15:40:07 system sshd[22647]: Starting session: shell on pts/2 for root from 173.161.228.229 port 47750",
]


class TestShadowDatabase(unittest.TestCase):

    @classmethod
    def setUpClass(cls) -> None:
        ssh_login_setup()

    @classmethod
    def tearDownClass(cls) -> None:
        Path(DATABASE_PATH).unlink(missing_ok=True)

    def test_message_parser(self):
        self.assertTrue(
            all([SSHLogin(m).is_interesting for m in interesting]),
            msg='parser missing interesting loglines'
        )
        self.assertTrue(
            all([not SSHLogin(m).is_interesting for m in uninteresting]),
            msg='parser saving uninteresting loglines'
        )

    def test_sshlogin_database(self):
        database = SSHLoginDatabase()

        for message in interesting:
            database.add_login(
                SSHLogin(message),
            )

        cursor = database.connection().execute(
            "select count(*) from ssh_login where success = 1"
        )
        self.assertEqual(
            cursor.fetchone()[0],
            2,
            msg='database must contain exactly two successful logins'
        )

        suspicious = database.suspicious_logins()
        self.assertEqual(
            len(suspicious),
            1,
            msg='database must contain only one suspicious login'
        )

        database.mark_reported(ids=[lgn['id'] for lgn in suspicious])

        suspicious = database.suspicious_logins()
        self.assertEqual(
            len(suspicious),
            0,
            msg='database must contain no suspicious logins'
        )


if __name__ == '__main__':
    unittest.main()
