from __future__ import annotations

import logging
import socket
from pathlib import Path
from threading import Thread

from watchdog.observers import Observer

from ..eventhandlers import BaseFileEventHandler, BaseSocketMessageHandler

logger = logging.getLogger(__name__)


def rm_socket(socket_path):
    try:
        Path(socket_path).unlink()
        logger.info('Deleting existing socket: %s' % socket_path)
    except FileNotFoundError:
        pass


class HedgeFileObserver(Observer):
    """
    Inherit from superclass that defines "watch_file"
    """
    EventHandler: BaseFileEventHandler = None

    def start(self) -> None:
        eventhandler = self.EventHandler()
        watch_file = eventhandler.watch_file
        watch_file_path = Path(watch_file)

        if not watch_file_path.exists():
            logger.error('Watch file not found: %s' % watch_file)
            return

        # Manually trigger the Actor's process method, in case
        # any events were missed while the hedge process wasn't
        # running. For example, the password might have changed
        # via Single-User boot mode.
        eventhandler.actor.process()

        logger.info('Starting observer for %s' % watch_file)

        self.schedule(eventhandler, watch_file, recursive=watch_file_path.is_dir())
        super().start()

    def stop(self) -> None:

        super().stop()
        logger.info('%s Observer stopped' % self.__class__.__name__)

        try:
            super().join(.25)
        except RuntimeError:
            logger.info('%s Observer was never started' % self.__class__.__name__)
            pass


class HedgeSocketObserver(Thread):
    """
    An object from this class represents
    a thread that listens to a unix socket
    """
    EventHandler: BaseSocketMessageHandler = None

    _socket: socket.socket = None
    _eventhandler: BaseSocketMessageHandler = None

    def __init__(self):
        super().__init__(target=self.observe_socket, daemon=True)

    def start(self) -> None:
        self._eventhandler = self.EventHandler()
        watch_socket = self._eventhandler.watch_socket

        rm_socket(watch_socket)
        logger.info('Starting observer for %s' % watch_socket)

        self._socket = socket.socket(socket.AF_UNIX, socket.SOCK_DGRAM)
        self._socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        self._socket.bind(watch_socket)

        super().start()

    def observe_socket(self) -> None:
        while True:
            try:
                self._eventhandler.new_message(message=self._socket.recv(4096))
            except Exception:  # noqa
                self.observe_socket()

    def stop(self) -> None:
        try:
            super().join(.25)
            self._socket.close()
            rm_socket(self._eventhandler.watch_socket)
            logger.info('%s Observer stopped' % self.__class__.__name__)
        except RuntimeError:
            logger.info('%s Observer was never started' % self.__class__.__name__)
            pass


__all__ = (
    'HedgeFileObserver',
    'HedgeSocketObserver',
)
