import logging

from .actors import shadow_file_passwords, ShadowDatabase, SSHLoginDatabase

logger = logging.getLogger(__name__)


def shadow_setup():
    """
    Setup shadow database.
    """
    database = ShadowDatabase()
    connection = database.connection()

    exists = "SELECT name FROM sqlite_master WHERE type='table' AND name='shadow';"
    cur = connection.execute(exists)
    table_exists = cur.fetchone()

    if table_exists:
        connection.close()
        return False

    table_sql = """
    CREATE TABLE shadow (
    id integer primary key,
    created datetime default current_timestamp,
    reported boolean default false,
    sp_namp text,
    sp_pwdp text,
    sp_lstchg int,
    sp_min int,
    sp_max int,
    sp_warn int,
    sp_inact int,
    sp_expire int,
    sp_flag int
    )
    """

    index_sql = "CREATE UNIQUE INDEX name_password_date_idx ON shadow(sp_namp, sp_pwdp, sp_lstchg)"

    try:
        connection.execute(table_sql)
        connection.commit()
        connection.execute(index_sql)
        connection.commit()
    except Exception as e:
        logger.exception(e)
        return False

    passwords = shadow_file_passwords()
    if not passwords:
        return True

    for pwd in passwords:
        pwd['reported'] = True

    database.add_passwords(passwords)
    connection.close()

    return True


def ssh_login_setup():
    """
    Setup SSH login database.
    """
    database = SSHLoginDatabase()
    connection = database.connection()

    exists = "SELECT name FROM sqlite_master WHERE type='table' AND name='ssh_login';"
    cur = connection.execute(exists)
    table_exists = cur.fetchone()

    if table_exists:
        connection.close()
        return False

    table_sql = """
    CREATE TABLE ssh_login (
    id integer primary key,
    timestamp datetime,
    username text,
    ip text,
    success boolean,
    authtype text,
    reported boolean default false
    )
    """

    index1_sql = "CREATE UNIQUE INDEX timestamp_username_idx ON ssh_login(timestamp, username)"
    index2_sql = "CREATE INDEX success_authtype_idx ON ssh_login(success, authtype)"

    try:
        connection.execute(table_sql)
        connection.commit()
        connection.execute(index1_sql)
        connection.execute(index2_sql)
        connection.commit()
    except Exception as e:
        logger.exception(e)
        return False

    connection.close()

    return True


__all__ = (
    'shadow_setup',
    'ssh_login_setup',
)
