from __future__ import annotations
import logging
import os
import signal
from watchdog.events import FileSystemEventHandler, FileSystemEvent
from ..actors import BaseActor

logger = logging.getLogger(__name__)


# ----------------------------------------------------------------------
class BaseFileEventHandler(FileSystemEventHandler):
    """
    Base event handler to subclass for all Hedge file handler cases.
    """
    watch_file = ''
    Actor: BaseActor = None

    def __init__(self):
        self.actor = self.Actor()

    def __str__(self):
        return self.__class__.__name__

    def on_closed(self, event: FileSystemEvent):
        self.terminate_handler(event)

    def on_deleted(self, event: FileSystemEvent):
        self.terminate_handler(event)

    def on_moved(self, event: FileSystemEvent):
        self.terminate_handler(event)

    def on_modified(self, event) -> None:
        """
        Fire an Actor on receiving the event of interest.
        """
        logger.info('Received event %s.' % event)

        self.actor.process()

    def terminate_handler(self, event: FileSystemEvent):
        """
        Catch events where we should restart.
        The task manager should restart to watch the new inode.
        """
        logger.info('Terminating - received event: %s' % event)
        os.kill(os.getgid(), signal.SIGTERM)


class BaseSocketMessageHandler:
    watch_socket = ''
    Actor: BaseActor = None

    def __init__(self):
        self.actor = self.Actor()

    def __str__(self):
        return self.__class__.__name__

    def new_message(self, message) -> None:
        self.actor.process(message)


__all__ = (
    'BaseFileEventHandler',
    'BaseSocketMessageHandler',
)
