from contextlib import closing
import logging
import sqlite3
from .settings import DATABASE_PATH

logger = logging.getLogger(__name__)


class HedgeDatabase:
    """
    General database for hedge library.
    """

    def connection(self):
        """
        Return a fresh connection each time.

        Context managers are used to handle connections like so:
        with closing(self.connection()) as con, con, closing(con.cursor()) as cur:

        The with variables perform as follows:
            * The first `con` auto-closes
            * The second `con` auto-commits
            * The `cur` auto-closes the cursor
        """
        connection = sqlite3.connect(DATABASE_PATH)
        connection.row_factory = sqlite3.Row

        return connection

    def fetchall(self, sql: str, params=None) -> list:
        """
        Execute a lookup SQL query and return results.
        """
        cur_params = (sql, params) if params else (sql,)

        try:
            with closing(self.connection()) as con, con, closing(con.cursor()) as cur:
                cur.execute(*cur_params)
                return cur.fetchall()
        except Exception as e:
            logger.exception(e)
            return []

    def insert_or_update(self, sql: str, params) -> bool:
        """
        Execute a query that inserts or updates a record.
        """
        try:
            with closing(self.connection()) as con, con, closing(con.cursor()) as cur:
                cur.execute(sql, params)
                return True
        except Exception as e:
            logger.exception(e)
            logger.error(f'Unable to execute {sql} with params {params}')

        return False

    def execute_many(self, sql: str, params) -> bool:
        """
        Execute a query that inserts or updates many records in a single statement.
        """
        try:
            with closing(self.connection()) as con, con, closing(con.cursor()) as cur:
                cur.executemany(sql, params)
                return True
        except Exception as e:
            logger.exception(e)
            logger.error(f'Unable to execute {sql} with params {params}')

        return False

    def __str__(self):
        return self.__class__.__name__


__all__ = [
    'HedgeDatabase',
]
