import logging
from pathlib import Path
from pyzabbix import ZabbixMetric, ZabbixSender
import socket

from .base import BaseAction

logger = logging.getLogger(__name__)


class ZabbixMetricAlert(BaseAction):
    """
    Send key/value metrics to Zabbix server.
    """
    key = ''

    def process(self):
        if not Path('/etc/zabbix/zabbix_agentd.conf').exists():
            return False

        hostname = socket.gethostname()

        try:
            username = self.record['sp_namp']
            zabbix_metrics = [ZabbixMetric(hostname, self.key, username)]
            ZabbixSender(use_config=True).send(zabbix_metrics)
        except Exception as e:
            logger.exception(e)
            return False

        logger.info(f'Completed {self.__class__.__name__} alert for {username}')
        return True


class ZabbixShadowAlert(ZabbixMetricAlert):
    """
    Send metric to Zabbix when a password is changed.
    """
    key = 'hedge.ssh.password.change'


class ZabbixLoginEmailAlert(ZabbixMetricAlert):
    """
    Send metric to Zabbix when a suspicious login occurs.
    """
    key = 'hedge.ssh.suspicious.login'


__all__ = (
    'ZabbixMetricAlert',
    'ZabbixShadowAlert',
    'ZabbixLoginEmailAlert',
)
