import logging
import os
import smtplib
import ssl
from email.message import EmailMessage

import certifi
from jinja2 import Template

import hedge
from .base import BaseAction
from ..settings import (
    ADMINURL,
    DEFAULT_FROM_EMAIL,
    EMAIL_HOST,
    EMAIL_PORT,
    EMAIL_HOST_USER,
    EMAIL_HOST_PASSWORD,
    EMAIL_RECIPIENT,
    SYSTEM_NAME,
    TEMPLATES,
)
from ..utils.dates import localize_time

os.environ['SSL_CERT_FILE'] = certifi.where()
logger = logging.getLogger(__name__)


class BaseEmailAlert(BaseAction):
    """
    Send email with incident details
    """
    email_subject = ''
    email_template = ''

    def get_context(self):
        """Return template context in subclass"""
        raise NotImplementedError

    def process(self):
        """
        Send email alert
        """
        template = self.load_template(self.email_template)
        if not template:
            logger.error(f'Unable to load template {self.email_template!r}')
            return False

        context = self.get_context()

        msg = EmailMessage()
        msg['From'] = DEFAULT_FROM_EMAIL
        msg['To'] = EMAIL_RECIPIENT
        msg['Subject'] = f'{self.email_subject} on {SYSTEM_NAME}'

        msg.set_content(template.render(**context), subtype='html')

        ssl_context = ssl.create_default_context()
        with smtplib.SMTP(EMAIL_HOST, port=EMAIL_PORT) as smtp_server:
            smtp_server.starttls(context=ssl_context)
            smtp_server.login(EMAIL_HOST_USER, EMAIL_HOST_PASSWORD)
            smtp_server.send_message(msg)

        logger.info(f'Completed {self.__class__.__name__} alert for {context}')
        return True

    @staticmethod
    def load_template(template):
        """
        Load html from templates directory, or extract from the egg
        """
        try:
            with open(f'{TEMPLATES}/{template}', 'r', encoding="utf-8") as tf:
                return Template(tf.read())
        except (FileNotFoundError, NotADirectoryError):
            pass

        template_file = f'hedge/templates/{template}'
        try:
            return Template(hedge.__loader__.get_data(template_file).decode('utf8'))
        except (FileNotFoundError, OSError) as e:
            logger.exception(e)
            print(f'{template_file} does not exist')
            return


class ShadowEmailAlert(BaseEmailAlert):
    email_subject = 'Password changed'
    email_template = 'password_changed.html'

    def get_context(self):
        return {
            'ADMINURL': ADMINURL,
            'user': self.record['sp_namp'],
            'created': localize_time(self.record['created']),
            'SYSTEM_NAME': SYSTEM_NAME,
        }


class SSHLoginEmailAlert(BaseEmailAlert):
    email_subject = 'Suspicious SSH activity'
    email_template = 'suspicious_login.html'

    def get_context(self):
        return {
            'ADMINURL': ADMINURL,
            'ip': self.record['ip'],
            'user': self.record['username'],
            'timestamp': self.record['timestamp'],
            'auth': self.record['authtype'].capitalize(),
            'SYSTEM_NAME': SYSTEM_NAME,
        }


__all__ = (
    'ShadowEmailAlert',
    'SSHLoginEmailAlert',
)
