from __future__ import annotations

import logging

from tabulate import tabulate

from .base import BaseActor
from ..alerts import SSHLoginEmailAlert, ZabbixLoginEmailAlert
from ..database import HedgeDatabase
from ..utils.rsyslog import SSHLogin

logger = logging.getLogger(__name__)


# ------------------------------------------------------------------------------
class SSHLoginActor(BaseActor):
    """
    Parse new rsyslog secure login message received through socket.
    """

    def process(self, message) -> None:
        """
        Parse new rsyslog secure login message and check for suspicious logins.
        """
        login = SSHLogin(message)
        if not login.is_interesting:
            return

        logger.info(login.message)
        database = SSHLoginDatabase()
        database.add_login(login)

        suspicious = database.suspicious_logins()
        if suspicious:
            self.alerts(logins=suspicious)
            database.mark_reported(ids=[lgn['id'] for lgn in suspicious])

    def alerts(self, logins) -> list:
        for login in logins:
            SSHLoginEmailAlert(record=login).process()
            ZabbixLoginEmailAlert(record=login).process()
        return logins


# ------------------------------------------------------------------------------
class SSHLoginDatabase(HedgeDatabase):
    """
    Add some helper methods to default Hedge database.
    """

    def add_login(self, login: SSHLogin) -> None:
        """
        Add new ssh login record.
        """
        sql = """
        INSERT INTO ssh_login(
            timestamp,
            username,
            ip,
            success,
            authtype
        ) VALUES (?, ?, ?, ?, ?)
        """
        params = (
            login.timestamp,
            login.username,
            login.ip,
            login.success,
            login.authtype,
        )

        self.insert_or_update(sql, params)

    def suspicious_logins(self) -> list:
        """
        Get suspicious-looking ssh logins.
        Criteria for suspicion:
         * Logged in with password
         * More later...
        """
        sql = """
        SELECT * FROM ssh_login WHERE
            authtype = 'password'
            AND
            success = true
            AND
            reported = false
        """
        return self.fetchall(sql)

    def mark_reported(self, ids: list[int]) -> None:
        """
        Mark all entries with specified record IDs as reported.
        """
        qps = ("?," * len(ids))[:-1]
        if self.insert_or_update(f"UPDATE ssh_login SET reported = true WHERE id in ({qps})", ids):
            logger.info(f'Marked IDs {ids} reported')

    def history(self) -> list:
        """
        Return all logins ever recorded.
        """
        return self.fetchall("SELECT * FROM ssh_login ORDER BY timestamp DESC")


# ------------------------------------------------------------------------------
def ssh_history() -> None:
    """
    Print out all SSH logins ever recorded.
    """
    tracked_logins = [('timestamp', 'username', 'ip', 'success', 'authtype')]

    for login in SSHLoginDatabase().history():
        tracked_logins.append((
            login['timestamp'],
            login['username'],
            login['ip'],
            login['success'] and True or False,
            login['authtype'].capitalize(),
        ))

    print('\nSSH login history')
    print(tabulate(tracked_logins, headers="firstrow"))
    print('\n')


__all__ = (
    'SSHLoginActor',
    'SSHLoginDatabase',
    'ssh_history',
)
