from __future__ import annotations

import hashlib
import logging
from datetime import datetime, timedelta

import spwd
from tabulate import tabulate

from .base import BaseActor
from ..alerts import ShadowEmailAlert, ZabbixShadowAlert
from ..database import HedgeDatabase
from ..settings import SHADOW_FIELDS
from ..utils.dates import localize_time

logger = logging.getLogger(__name__)


# ------------------------------------------------------------------------------
class ShadowActor(BaseActor):
    """
    Parse /etc/shadow after receiving notice that the file has been modified.
    """

    def process(self) -> None:
        """
        Read /etc/shadow and check for changed passwords.
        """
        database = ShadowDatabase()
        database.add_passwords(passwords=shadow_file_passwords())

        changes = database.unreported_changes()
        if changes:
            self.alerts(passwords=changes)
            database.mark_reported(ids=[pwd['id'] for pwd in changes])

    def alerts(self, passwords) -> list:
        for password in passwords:
            ShadowEmailAlert(record=password).process()
            ZabbixShadowAlert(record=password).process()
        return passwords


# ------------------------------------------------------------------------------
class ShadowDatabase(HedgeDatabase):
    """
    Add some helper methods to default Hedge database.
    """

    def add_passwords(self, passwords: list[dict]) -> None:
        """
        Add new password changes from /etc/shadow.
        """
        connection = self.connection()

        new = []
        for pwd in passwords:
            cursor = connection.execute(
                "SELECT * FROM shadow WHERE sp_namp = ? AND sp_lstchg = ? AND sp_pwdp = ?",
                (pwd['sp_namp'], pwd['sp_lstchg'], pwd['sp_pwdp'])
            )
            if not cursor.fetchone():
                new.append(pwd)

        connection.close()
        if not new:
            return

        fields = ", ".join(new[0].keys())
        attributes = ','.join([f':{key}' for key in new[0]]).strip(',')

        if self.execute_many(f"INSERT INTO shadow({fields}) VALUES ({attributes})", new):
            logger.info(f'Added password changes from /etc/shadow')

    def unreported_changes(self) -> list:
        """
        Get all entries that haven't been reported.
        """
        return self.fetchall("SELECT * FROM shadow WHERE reported = false")

    def mark_reported(self, ids: list[int]) -> None:
        """
        Mark all entries with specified record IDs as reported.
        """
        qps = ("?," * len(ids))[:-1]

        if self.insert_or_update(f"UPDATE shadow SET reported = true WHERE id IN ({qps})", ids):
            logger.info(f'Marked {ids} as reported')

    def users(self) -> list:
        """
        Return latest password entry of all users tracked in the database.
        """
        sql = """
            SELECT * 
              FROM shadow AS t1
              INNER JOIN (SELECT MAX(id) AS id FROM shadow GROUP BY sp_namp) AS t2
              ON t1.id = t2.id
              ORDER BY id DESC;
        """
        return self.fetchall(sql)

    def history(self) -> list:
        """
        Return all password changes ever recorded.
        """
        return self.fetchall("SELECT * FROM shadow ORDER BY created DESC")


# ------------------------------------------------------------------------------
def shadow_file_passwords() -> list[dict]:
    """
    Return all users from /etc/shadow that have passwords assigned.
    """
    pwds = [pwd for pwd in spwd.getspall() if pwd.sp_pwdp not in ('*', '!!')]
    passwords = []

    for pwd in pwds:
        data = {}
        for field in SHADOW_FIELDS:
            data[field] = getattr(pwd, field)
        passwords.append(data)

    for pwd in passwords:
        pwd_hash = pwd['sp_pwdp']
        pwd['sp_pwdp'] = hashlib.blake2s(pwd_hash.encode('utf8')).hexdigest()

    return passwords


# ------------------------------------------------------------------------------
def system_users() -> None:
    """
    Print out all users that can login to the system.
    """
    epoch = datetime(year=1970, day=1, month=1)
    users = [('name', 'password date')]
    tracked_users = [('name', 'password date')]

    for user in shadow_file_passwords():
        users.append((
            user['sp_namp'],
            epoch + timedelta(days=user['sp_lstchg']),
        ))

    for user in ShadowDatabase().users():
        tracked_users.append((
            user['sp_namp'],
            epoch + timedelta(days=user['sp_lstchg']),
        ))

    print('\nSystem users')
    print(tabulate(users, headers="firstrow"))

    print('\nTracked users')
    print(tabulate(tracked_users, headers="firstrow"))
    print('\n')


# ------------------------------------------------------------------------------
def shadow_history() -> None:
    """
    Print out all password changes ever recorded.
    """
    tracked_users = [('name', 'password date')]

    for user in ShadowDatabase().history():
        tracked_users.append((
            user['sp_namp'],
            localize_time(user['created']),
        ))

    print('\nPassword change history')
    print(tabulate(tracked_users, headers="firstrow"))
    print('\n')


__all__ = (
    'ShadowActor',
    'ShadowDatabase',
    'shadow_file_passwords',
    'system_users',
    'shadow_history',
)
