import logging
import sys
import time
import signal

from .actors import shadow_history, system_users, ssh_history
from .observers import (
    ShadowObserver,
    SSHLoginObserver,
)
from .installation import shadow_setup, ssh_login_setup

logger = logging.getLogger('hedge')


def setup():
    """
    Setup and configure the Hedge database.
    """
    shadow_setup()
    ssh_login_setup()


def run():
    """
    Run Hedge process.
    """

    class ProcessExit(Exception):
        """
        Custom exception which is used to trigger the clean exit
        of all running threads and the main program.
        """
        pass

    def service_shutdown(signum, frame):
        logger.info('Caught signal %d' % signum)
        raise ProcessExit

    signal.signal(signal.SIGTERM, service_shutdown)
    signal.signal(signal.SIGINT, service_shutdown)
    signal.signal(signal.SIGHUP, service_shutdown)

    all_observers = []

    try:
        for observer in (ShadowObserver, SSHLoginObserver):
            all_observers.append(observer())

        for observer in all_observers:
            logger.info(f'starting observer {observer}')
            observer.start()

        # Keep the main thread running, otherwise signals are ignored.
        try:
            while True:
                time.sleep(1.0)
        except (ProcessExit, KeyboardInterrupt):
            pass

    except Exception as e:
        logger.exception(e)

    logger.info('Shutdown Hedge observers')
    for observer in all_observers:
        try:
            observer.stop()
        except (ProcessExit, KeyboardInterrupt):
            pass


def print_help():
    print(f"usage: hedge [<command>]\n")
    print("Valid commands are:")
    for name, func in COMMANDS.items():
        docs = ' ' * (15 - len(name)) + func.__doc__.strip()
        print(f'    {name} {docs}')


COMMANDS = {
    'users': system_users,
    'shadow_history': shadow_history,
    'ssh_history': ssh_history,
    'run': run,
    'setup': setup,
}


def main():

    if len(sys.argv) <= 1:
        print_help()
        sys.exit(0)

    hedge_function = COMMANDS.get(sys.argv[1], print_help)
    hedge_function()


if __name__ == "__main__":
    main()
