#!/usr/bin/python3
import distro
import io
import re
import os
from os.path import dirname, join
import sys
from setuptools import setup

v = sys.version_info


def read(*names, **kwargs):
    return io.open(join(dirname(__file__), *names), encoding=kwargs.get('encoding', 'utf8')).read()


def get_data_files():
    IS_DRAWBRIDGE_OS = distro.name().lower() in ('drawbridgeos', 'vyos', 'ubuntu')
    data_files = [
        ('/etc/supervisord.d', [os.path.join('config/supervisord', 'glade.conf')]),
        ('/etc/logrotate.d', [os.path.join('config/logrotate', 'glade')]),
    ]
    if IS_DRAWBRIDGE_OS:
        data_files.append(
            ('/usr/lib/systemd/systemd', [os.path.join('config/systemd', 'glade.service')])
        )
    else:
        data_files.append(
            ('/etc/supervisord.d', [os.path.join('config/supervisord', 'glade.conf')])
        )

    return data_files


def get_version():
    with open('glade/__init__.py', 'r') as f:
        line = f.readline()
        match = re.match(r'__version__ = \'([\d\.]+)\'', line)

        if not match:
            raise ImportError("Can't read the version of glade")

        version = match.group(1)
        return version


def get_packages(package):
    """
    Return root package and all sub-packages.
    """
    return [
        dirpath for dirpath, dirnames, filenames in os.walk(package)
        if os.path.exists(os.path.join(dirpath, '__init__.py'))
    ]


setup(
    name='glade',
    version=get_version(),
    license='SAP',
    description='Extract device information from local network',
    author='Dave Burkholder',
    author_email='dave@compassfoundation.io',
    url='https://code.compassfoundation.io/dave/glade',
    packages=get_packages('glade'),
    data_files=get_data_files(),
    zip_safe=True,
    options={
        'bdist_rpm': {
            'post_install': 'post_install.sh',
            'pre_install': 'pre_install.sh',
        },
    },
    classifiers=[
        'Intended Audience :: Developers',
        'License :: SAP',
        'Operating System :: Unix',
        'Operating System :: POSIX',
        'Programming Language :: Python :: 3.9',
        'Programming Language :: Python :: Implementation :: CPython',
        'Programming Language :: Python :: Implementation :: PyPy',
        'Topic :: Utilities',
    ],
    keywords=[
        'Security Appliance',
        'Log Cabin',
    ],
    install_requires=[
        'distro',
        'lcconfig',
        'lcutils',
        'pysmb',
        'zeroconf',
        'django',
        'dnspython',
    ],
    entry_points={"console_scripts": [
        "glade = glade.__main__:main",
    ]},
)
