import os
import pickle
from queue import Queue
import signal
from socketserver import (
    StreamRequestHandler,
    TCPServer,
)
import time
from threading import Event, Thread

from glade.extractors import (
    check_device_type,
    check_hostname,
    check_platform,
)

DEVICE_CACHE = {}


class GladeRequestHandler(StreamRequestHandler):

    def persist(self, data: dict) -> None:
        """
        Override on subclass to handle persisting
        data to file system or database.
        """
        pass

    def handle(self):
        """
        Get device details from listeners, and cache results.
        If results haven't changed from last run, call task
        to update LocalDevice data in Log Cabin console.

        Message payload will come from listeners like so:
        {
            'source': 'zeroconf',
            'type': 'printer',
            'hostname': 'EpsonWorkforce-7510',
            'mac': 'f0:99:b6:05:dc:e9',
            'ip': '192.168.40.249',
        }
        """
        try:
            data = pickle.loads(self.rfile.readline())
        except pickle.UnpicklingError:
            return

        try:
            ip = data['ip']
            if ip == '0.0.0.0':
                return

            hostname = check_hostname(hostname=data['hostname'].lower(), ip=ip)
            if DEVICE_CACHE.get(ip, f'None for {ip}!') == hostname:
                return

        except (KeyError, TypeError):
            return

        DEVICE_CACHE[ip] = hostname
        platform = check_platform(hostname=hostname, ip=ip, mac=data['mac'])
        data.update({
            'platform':
            platform,
            'device_type':
            check_device_type(hostname=hostname, mac=data['mac'], platform=platform),
            'task_name':
            f'Update {hostname} / {ip} details from Glade',
        })
        self.persist(data=data)


class GladePoolMixin:
    """
    Mixin to use a fixed pool of threads to handle requests,
    instead of launching one thread per request. That would
    overwhelm max database connections when processing a
    large arp table.
    """

    pool_size = 10
    timeout_on_get = 0.5

    def __init__(self):
        self.request_queue = Queue(self.pool_size)
        self.shutdown_flag = Event()
        self.threads = []

        for _ in range(self.pool_size):
            self.threads.append(Thread(target=self.process_request_thread))

        for thread in self.threads:
            thread.start()

    def process_request_thread(self):
        """Same as in BaseServer, but as a thread."""
        while not self.shutdown_flag.is_set():
            if self.request_queue.empty():
                time.sleep(.5)
                continue
            request, client_address = self.request_queue.get(
                timeout=self.timeout_on_get,
            )
            try:
                self.finish_request(request, client_address)
                self.shutdown_request(request)
            except Exception:
                self.handle_error(request, client_address)
                self.shutdown_request(request)
            self.request_queue.task_done()

    def process_request(self, request, client_address):
        """Queue the given request."""
        self.request_queue.put((request, client_address))

    def shutdown(self):
        """
        Instantly stop handling events
        """
        self.shutdown_flag.set()

        for thread in self.threads:
            thread.join()

        # kill the process here to avoid deadlocking
        os.kill(os.getpid(), signal.SIGHUP)


class GladeServer(GladePoolMixin, TCPServer):
    """
    Socket Server is where listeners post network services / broadcast events.

    Data should be maintained as Local Devices in Log Cabin console.
    """
    allow_reuse_address = True

    def __init__(self, server_address, RequestHandlerClass, bind_and_activate=True):
        GladePoolMixin.__init__(self)
        TCPServer.__init__(self, server_address, RequestHandlerClass, bind_and_activate)


class GladeExit(KeyboardInterrupt):
    """
    Trigger the clean exit of all running threads and the main program.
    """
    pass


def service_shutdown(signum, frame):
    print(f'Caught signal {signum}')
    raise GladeExit


__all__ = (
    'GladeExit',
    'GladePoolMixin',
    'GladeRequestHandler',
    'GladeServer',
    'service_shutdown',
)
