from multiprocessing.pool import ThreadPool
import socket
import subprocess
from typing import Dict, List


def host_reachable(host: str) -> bool:
    """
    Returns True if host (str) responds to a ping request.
    Remember that a host may not respond to a ping (ICMP) request even if the host name is valid.
    """
    if not host:
        return False

    command = ['sudo', 'ping', '-c', '1', '-w', '1', host]
    return subprocess.call(command) == 0


def connect(host: str, port: int):
    """
    Connect to port on given host, tracking open / closed ports.
    """
    sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    sock.settimeout(0.25)

    try:
        sock.connect((host, port))
        status = 'open'
    except Exception:
        status = 'closed'

    sock.close()

    return status, port


def port_scan(host: str, start: int = 1, end: int = 65535) -> Dict[str, List[str]]:
    """
    Do port scan on host / IP address
    """

    ports = {'open': [], 'closed': []}
    port_scans = [(host, port) for port in range(start, end + 1)]
    pool_size = len(port_scans) if len(port_scans) <= 7000 else 7000

    with ThreadPool(pool_size) as pool:
        results = pool.starmap(connect, port_scans)
        [ports[status].append(port) for status, port in results]

    return ports


__all__ = [
    'host_reachable',
    'port_scan',
]
