from lcutils import get_oui_vendor
import socket


def port_open(port: int, ip: str) -> bool:
    """
    Check if port is open on the network
    """

    try:
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        sock.connect((ip, port))
        is_open = True
    except Exception:
        is_open = False
    finally:
        sock.close()

    return is_open


def is_android(hostname: str) -> bool:
    """
    Device at IP is running Android
    """
    return hostname.startswith('android-')


def is_macintosh(hostname: str, mac: str) -> bool:
    """
    Device at IP is Macintosh computer.
    Should be checked after iOS checks.
    """
    if get_oui_vendor(mac).startswith('Apple'):
        return True
    return hostname.endswith('-mbp')


def is_ios(ip: str) -> bool:
    """
    Device at IP is iOS mobile device
    """
    return port_open(port=62078, ip=ip)


def is_windows(ip: str) -> bool:
    """
    Device at IP is Windows Computer
    """
    return port_open(port=139, ip=ip)


def check_platform(hostname: str, ip: str, mac: str) -> str:
    """
    Try to determine platform of device.
    """
    if is_android(hostname=hostname):
        return 'Android'

    if is_ios(ip=ip):
        if 'ipad' in hostname:
            return 'iPad'
        if 'ipod' in hostname:
            return 'iPod'
        if 'iPhone' in hostname:
            return 'iPhone'
        return 'UNK'

    if is_macintosh(hostname=hostname, mac=mac):
        return 'Macintosh'

    if is_windows(ip=ip):
        return 'Windows'

    return 'UNK'


__all__ = [
    'check_platform',
]
