from dns import (
    exception,
    resolver,
    reversename,
    rdatatype,
)
from nmb.NetBIOS import NetBIOS
from ..settings import DETECT_HOSTNAME, DIRECTORY_SERVER


def netbios_device_name(ip: str) -> str:
    """
    Get netbios device name if available.
    """
    nb = NetBIOS()

    name = nb.queryIPForName(ip=ip, timeout=2.5)

    try:
        # query returns list - ['SHOP-OFFICE'], etc
        return name[0].lower()
    except TypeError:
        return ''


def lookup_hostname(hostname: str, nameservers=None, configure=False) -> str:
    """
    Lookup a hostname, and optionally specify name servers.
    """
    if not nameservers:
        nameservers = [DIRECTORY_SERVER] if DIRECTORY_SERVER else None

    configure = True if not nameservers else configure
    resolver.default_resolver = resolver.Resolver(configure=configure)

    if nameservers:
        resolver.default_resolver.nameservers = nameservers

    try:
        for answer in resolver.resolve(hostname.strip(), rdatatype.A):
            return str(answer)
    except exception.DNSException:
        return ''


def reverse_lookup_address(ip: str, nameservers=None, configure=False) -> str:
    """
    When another server is providing DNS service, usually Windows AD server,
    perform reverse DNS lookup to retrieve hostname. Preferable to netbios
    as netbios will fail if computer is Off/Sleeping or firewall is in
    public mode.
    """
    if not nameservers:
        nameservers = [DIRECTORY_SERVER] if DIRECTORY_SERVER else None

    configure = True if not nameservers else configure
    resolver.default_resolver = resolver.Resolver(configure=configure)

    if nameservers:
        resolver.default_resolver.nameservers = nameservers

    try:
        iplookup = reversename.from_address(ip.strip())
        dns_name = resolver.resolve(iplookup, rdatatype.PTR)[0]
    except exception.DNSException:
        return ''

    try:
        return dns_name.to_text().split('.')[0]
    except IndexError as e:
        print(e)
        return str(dns_name)


def check_hostname(hostname: str, ip: str) -> str:
    """
    If hostname is not set or is wildcard, attempt
    to retrieve it through reverse dns lookup or netbios.

    Reverse DNS lookup is preferred, but requires that the
    DHCP server support reverse DNS lookups. Windows Servers
    support this but may not be setup.
    """
    if hostname and hostname != '*' \
            and hostname != ip \
            and not hostname.startswith(DETECT_HOSTNAME):
        return hostname

    for detector in (reverse_lookup_address, netbios_device_name):
        name = detector(ip)
        if name:
            return name

    return hostname


reverse_dns_lookup = reverse_lookup_address

__all__ = (
    'check_hostname',
    'netbios_device_name',
    'lookup_hostname',
    'reverse_lookup_address',
    'reverse_dns_lookup',
)
