from lcutils import NetworkInfo
import logging
import logging.config
import signal
import socket
import time
from threading import Thread

from zeroconf import (
    ServiceBrowser,
    Zeroconf,
)

from glade.listeners import (
    ARPTable,
    DHCPInterface,
    ServiceListener,
)
from glade.server import (
    GladeExit,
    GladeRequestHandler as GladeRequestHandlerBase,
    GladeServer,
    service_shutdown,
)
from .settings import portal_app, LOGGING


def main():
    logging.config.dictConfig(LOGGING)

    logger = logging.getLogger(__name__)
    s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    glade_port = 10200

    try:
        s.connect(('localhost', glade_port))
        logger.error('Glade is already running')
        raise SystemExit('Glade is already running')
    except Exception:  # noqa
        pass

    from glade.settings import console_config

    CLOUD_SERVER = console_config.get('cloud_server', True)
    FILTER_SERVER = console_config.get('filter_server', True)

    class GladeRequestHandler(GladeRequestHandlerBase):

        def persist(self, data):
            if not portal_app:
                logger.error('Scheduler is not configured')
                return

            portal_app.send_task(
                'devices.tasks.localdevices.update_localdevice_info',
                kwargs=data,
            )

    signal.signal(signal.SIGTERM, service_shutdown)
    signal.signal(signal.SIGINT, service_shutdown)

    zeroconf = Zeroconf()
    zeroconf_listener = ServiceListener()

    all_threads = []
    server = server_thread = None

    try:
        server = GladeServer(('localhost', glade_port), GladeRequestHandler)
        server_thread = Thread(target=server.serve_forever)
        server_thread.start()
        logger.info(f'GladeServer running in thread {server_thread.getName()}')

        # Get DHCP broadcast events on all local interfaces
        if FILTER_SERVER and not CLOUD_SERVER:
            for interface in NetworkInfo().lan_interfaces():
                all_threads.append(DHCPInterface(interface=interface))

        # Get printers and workstations advertising themselves on the network
        if FILTER_SERVER and not CLOUD_SERVER:
            for service in ('_workstation._tcp.local.', '_printer._tcp.local.'):
                ServiceBrowser(zeroconf, service, zeroconf_listener)

        if FILTER_SERVER and not CLOUD_SERVER:
            all_threads.append(ARPTable())

        for thred in all_threads:
            thred.start()

        # Keep the main thread running, otherwise signals are ignored.
        try:
            while True:
                time.sleep(1.0)
        except (GladeExit, KeyboardInterrupt):
            pass

    except Exception as e:
        logger.exception(e)

    # Set the shutdown flag on each thread to trigger a clean shutdown of each thread.
    logger.info('Shutdown Glade Listeners')
    for thread in all_threads:
        try:
            thread.shutdown_flag.set()
        except (GladeExit, KeyboardInterrupt):
            pass

    for thread in all_threads:
        try:
            thread.join(.25)
        except (GladeExit, KeyboardInterrupt):
            pass

    zeroconf.close()

    if server:
        logger.info('Shutdown Glade Server')
        server_thread.join(timeout=0.25)
        server.shutdown()


if __name__ == '__main__':
    main()
