import unittest

from ...attacks.homoglyphs import Attack, FraudCheck
from ..domain_spoofs import check_domain_spoofs


class DomainSpoofDetection(unittest.TestCase):

    def test_no_homoglyph_spoofs_detected(self):
        data = {
            'webpage_url': 'https://drawbridgetest.com/examples/homoglyph_urls.html',
            'external_hrefs': {
                'apple.com': {
                    'apple.com': ['apple.com'],
                },
                'paypal.com': {
                    'paypal.com': ['paypal.com'],
                },
                'google.com': {
                    'google.com': ['google.com'],
                },
            },
        }
        self.assertDictEqual(check_domain_spoofs(data), {})

    def test_homoglyph_spoofs_detected(self):
        data = {
            'webpage_url': 'https://drawbridgetest.com/examples/homoglyph_urls.html',
            'external_hrefs': {
                'aррle.com': {
                    'aррle.com': ['aррle.com'],
                },
                'pаypаl.com': {
                    'pаypаl.com': ['pаypаl.com'],
                },
                'ɡooɡle.com': {
                    'ɡooɡle.com': ['ɡooɡle.com'],
                },
            },
        }

        result = check_domain_spoofs(data)

        expected_result = {
            'aррle.com': Attack(name=FraudCheck.Unicode, targeted_domain='apple.com'),
            'pаypаl.com': Attack(name=FraudCheck.Unicode, targeted_domain='paypal.com'),
            'ɡooɡle.com': Attack(name=FraudCheck.Unicode, targeted_domain='google.com'),
        }
        self.assertDictEqual(
            result,
            expected_result,
            msg='Check result must match expected result',
        )

    def test_many_unicode_blocks_detected(self):
        data = {
            'webpage_url': 'https://funnylibrary.com',
            'external_hrefs': {
                'gooԁreаԁѕ.biz': {
                    'gooԁreаԁѕ.biz': ['gooԁreаԁѕ.com.biz'],
                },
            },
        }
        expected_result = {
            'gooԁreаԁѕ.biz': Attack(name=FraudCheck.UnicodeBlock, targeted_domain='gooԁreаԁѕ.biz'),
        }
        self.assertDictEqual(check_domain_spoofs(data), expected_result)


if __name__ == '__main__':
    unittest.main()
