from ..attacks.homoglyphs import (
    Attack,
    FraudCheck,
    check_unicode_homoglyph_attack,
    check_ascii_attack,
    check_punycode,
    check_unicode_blocks,
    perform_fraud_check,
)


def check_domain_spoofs(data: dict) -> dict[str, dict[str, Attack]]:
    """
    Expects a Redwood Starlark POST body with an `external_hrefs`
    key and inspects the domains for spoofing attacks of various types.

    data = {
        'webpage_url': 'https://drawbridgetest.com/examples/homoglyph_urls.html',
        'external_hrefs': {
            'aррle.com': {
                'aррle.com': ['aррle.com'],
            },
            'pаypаl.com': {
                'pаypаl.com': ['pаypаl.com'],
             },
            'ɡooɡle.com': {
                'ɡooɡle.com': ['ɡooɡle.com'],
            },
        },
    }
    """
    results = {}

    for domain in data['external_hrefs'].keys():

        if domain.isascii():
            if attack := perform_fraud_check(check_ascii_attack, domain):
                results[domain] = attack
            if attack := perform_fraud_check(check_punycode, domain):
                results[domain] = attack
            continue

        if attack := perform_fraud_check(check_unicode_homoglyph_attack, domain):
            results[domain] = attack

        # If a domain name includes too many blocks, skip the
        # is_popular_domain check, since it's likely up to no good.
        elif (attack := check_unicode_blocks(domain))['name'] != FraudCheck.NoMatch:
            results[domain] = attack

    return {'malicious/spoofed_domains': results} if results else results
