from invoke import task, run, UnexpectedExit
from lchttp import yaml_safe_load
from lcrsync import run as rsync
from lcrsync.exceptions import RemotesError, RsyncError
from pathlib import Path
import sys
from ..config_files import ConfigWriter, Retriever
from dalmatian.settings import (
    DalmatianConfig,
    DEFAULT_PUBLISHER_URL,
    IS_DRAWBRIDGE_OS,
    IPSET_DIR,
    PORTAL_ROOT,
    RPZ_DIR,
    SUBSCRIPTIONS_DIR,
    logging,
)

logger = logging.getLogger(__name__)

RSYNC_PWD_FILE = f'{PORTAL_ROOT}/privatesettings/.rpz.txt' \
    if IS_DRAWBRIDGE_OS else '/opt/sap/scripts/rpz.txt'


@task
def subscriber(ctx):
    """
    Pull updates from Publisher for all RPZ Zones, IPSets and Redwood Categories.
    """
    try:
        run('rpm -q pdns-recursor')
        recursor_installed = True
    except UnexpectedExit:
        recursor_installed = False

    cfg = DalmatianConfig().as_typed_dict()
    publisher_url = cfg.get('publisher_url', DEFAULT_PUBLISHER_URL)
    password_file = cfg.get('password_file', RSYNC_PWD_FILE)

    try:
        zones = rsync(
            source=f'rsync://rpz@{publisher_url}/rpz/',
            destination=f'{RPZ_DIR}/',
            options=['-rzai', f'--password-file={password_file}'],
        )

        ipsets = rsync(
            source=f'rsync://rpz@{publisher_url}/ipset/',
            destination=f'{IPSET_DIR}',
            options=['-rzai', f'--password-file={password_file}'],
        )
    except (RemotesError, RsyncError) as e:
        logger.exception(f'Unable to update subscriber: {e}')
        print('Unable to update subscriber')
        return

    if recursor_installed and zones and zones.get('stdout'):
        run('service pdns-recursor restart')

    # TODO write script to reload IPSets


@task
def publisher(ctx):
    """
    Update all upstream subscriptions to this machine for serving
    to Subscriber machines via Rsync. Only to be run if this system
    is then the Dalmatian publisher to all Compass subscriber systems.
    """
    cfg = DalmatianConfig().as_typed_dict('DALMATIAN')
    if not cfg.get('is_publisher', False):
        logger.info('Skipping updates; not a Publisher.')
        return

    for subscription in Path(SUBSCRIPTIONS_DIR).iterdir():
        update_subscription(f'{subscription}/settings.yml')


@task(pre=[publisher, subscriber])
def pubsub(ctx):
    """
    Process both the Publisher and Subscriber updates
    """
    pass


@task
def console(ctx):
    """
    Create DNS Firewall RPZ Zones in the Log Cabin Console from Dalmatian subscriptions.
    """
    pv = sys.version_info
    run(f'/usr/bin/python{pv.major}.{pv.minor} {PORTAL_ROOT}/manage.py add_rpzs')


def update_subscription(path):
    """
    Update subscription files from upstream
    """
    try:
        with open(path) as f:
            settings = yaml_safe_load(f.read())
    except FileNotFoundError:
        logger.error('%s not found' % path)
        return False

    RetrieverClass = Retriever.getClass(origin=settings['origin'])
    RetrieverClass(**settings).update()

    ConfigWriterClass = ConfigWriter.getClass(origin=settings['origin'])
    ConfigWriterClass(**settings).save()

    logger.info(f'{path} updated to latest settings')

    return True
