import unittest

from ..homoglyphs import (
    check_unicode_blocks,
    check_unicode_homoglyph_attack,
    check_tld_spoof,
    check_punycode,
    is_popular_domain,
    unicode_blocks,
    check_ascii_attack,
    FraudCheck,
)


class UrlSpoofTesting(unittest.TestCase):

    def test_unicode(self):
        for domain, answer in (
            ('bińg.com', FraudCheck.Unicode),
            ('раɣрɑǀ.com', FraudCheck.Unicode),
            ('ɡ၀၀ɡ𐌉ҽ.com', FraudCheck.Unicode),
            ('сha𐑈e.com', FraudCheck.Unicode),
            ('ԁısс၀⋁er.com', FraudCheck.Unicode),
            ('аmɑzဝn.com', FraudCheck.Unicode),
            ('whаtsаρρ.com', FraudCheck.Unicode),
            ('sр၀tify.com', FraudCheck.Unicode),
            ('nytımeѕ.com', FraudCheck.Unicode),
            ('ᖯinɡ.com', FraudCheck.Unicode),
            ('sհ၀ріfү.com', FraudCheck.Unicode),
            ('teӀ℮ɡгam.com', FraudCheck.Unicode),
            ('ebаγ.com', FraudCheck.Unicode),
            ('twᎥtсհ.tv', FraudCheck.Unicode),
            ('ԁısс၀rԁ.gg', FraudCheck.Unicode),
            ('уe׀р.com', FraudCheck.Unicode),
            ('gooԁreаԁѕ.com', FraudCheck.Unicode),
            ('𐑈ɑmѕυng.com', FraudCheck.Unicode),
            ('ınᏧеeԁ.com', FraudCheck.Unicode),
            ('асаdemᎥα.edu', FraudCheck.Unicode),
            ('сısс၀.com', FraudCheck.Unicode),
            ('f၀×news.com', FraudCheck.Unicode),
            ('newу၀rk℮r.com', FraudCheck.Unicode),
            ('уeاp.com', FraudCheck.Unicode),
            ('руtհ၀n.org', FraudCheck.Unicode),
            ('аոԁr၀Ꭵᑯ.com', FraudCheck.Unicode),
            ('stаte.gov', FraudCheck.Unicode),
            ('bing.com', FraudCheck.NoMatch),
        ):
            self.assertEqual(
                check_unicode_homoglyph_attack(domain)['name'],
                answer,
                f'{domain} answer {answer} is incorrect',
            )

    def test_check_punycode(self):
        for domain, answer in (
            ('xn--big-qwa.com', FraudCheck.Punycode),  # 'bińg.com'
            ('xn--fja30axby1ijcc.com', FraudCheck.Punycode),  # 'раɣрɑǀ.com'
            ('xn--hae-3ed13883a.com', FraudCheck.Punycode),  # 'сha𐑈e.com'
            ('xn--ser-iua530b36a599b007a.com', FraudCheck.Punycode),  # 'ԁısс၀⋁er.com'
            ('xn--mzn-gsb70vqt2a.com', FraudCheck.Punycode),  # 'аmɑzဝn.com'
            ('xn--whts-eoda78ec.com', FraudCheck.Punycode),  # 'whаtsаρρ.com'
            ('xn--stify-vye816s.com', FraudCheck.Punycode),  # 'sр၀tify.com'
            ('xn--nytme-p4a447c.com', FraudCheck.Punycode),  # 'nytımeѕ.com'
            ('xn--in-kgb978q.com', FraudCheck.Punycode),  # 'ᖯinɡ.com'
            ('xn--sf-lmc2g0uh1bvz8b.com', FraudCheck.Punycode),  # 'sհ၀ріfү.com'
            ('xn--team-17b01zf5ae62m.com', FraudCheck.Punycode),  # 'teӀ℮ɡгam.com'
            ('xn--eb-j9b31c.com', FraudCheck.Punycode),  # 'ebаγ.com'
            ('xn--twt-6ed92m2723c.tv', FraudCheck.Punycode),  # 'twᎥtсհ.tv'
            ('xn--sr-gpa152am1aea562r.com', FraudCheck.Punycode),  # 'ԁısс၀rԁ.dd'
            ('xn--goore-8ve3x22cda.com', FraudCheck.Punycode),  # 'gooԁreаԁѕ.com'
            ('xn--mng-fsb57pxyas812k.com', FraudCheck.Punycode),  # '𐑈ɑmѕυng.com'
            ('xn--ne-gpa302ag3af867d.com', FraudCheck.Punycode),  # 'ınᏧеeԁ.com'
            ('xn--dem-qxc54da2j2876m.edu', FraudCheck.Punycode),  # 'асаdemᎥα.edu'
            ('xn--s-eka56uca964m.com', FraudCheck.Punycode),  # 'сısс၀.com'
            ('xn--fnews-7oa827y.com', FraudCheck.Punycode),  # 'f၀×news.com'
            ('xn--newrkr-yrf666vs08a.com', FraudCheck.Punycode),  # 'newу၀rk℮r.com'
            ('xn--tn-kmcm02ms06a.org', FraudCheck.Punycode),  # 'руtհ၀n.org'
            ('xn--r-7sb14d1oh95bv2fnk11a.com', FraudCheck.Punycode),  # 'аոԁr၀Ꭵᑯ.com'
            ('bing.com', FraudCheck.NoMatch),
            ('google.com', FraudCheck.NoMatch),
        ):
            self.assertEqual(
                check_punycode(domain)['name'],
                answer,
                f'{domain} answer {answer} is incorrect',
            )

    def test_tld_spoof(self):
        for domain, answer in (
            ('googlecom.com', FraudCheck.TldSpoof),
            ('apple.com.org', FraudCheck.TldSpoof),
            ('bing.com.in', FraudCheck.TldSpoof),
            ('chase.com.com', FraudCheck.TldSpoof),
            ('chaseorg.com', FraudCheck.TldSpoof),
            ('youtubeio.com', FraudCheck.TldSpoof),
            ('shopify.site.com', FraudCheck.TldSpoof),
            ('academia.page.edu', FraudCheck.TldSpoof),
            ('discord.ie.com', FraudCheck.TldSpoof),
            ('discord.net.com', FraudCheck.TldSpoof),
            ('yelp.com.de', FraudCheck.TldSpoof),
            ('samsung.in.com', FraudCheck.TldSpoof),
            ('cisco.site.com', FraudCheck.TldSpoof),
            ('abebooks.to.com', FraudCheck.TldSpoof),
            ('pythonorg.com', FraudCheck.TldSpoof),
            ('androidorg.com', FraudCheck.TldSpoof),
            ('twitch.tv.com', FraudCheck.TldSpoof),
            ('discoverorg.com', FraudCheck.TldSpoof),
            ('indeedorg.com', FraudCheck.TldSpoof),
            ('google.site.com', FraudCheck.TldSpoof),
            ('amazon.com.site', FraudCheck.TldSpoof),
            ('microsoft.com', FraudCheck.NoMatch),
        ):
            self.assertEqual(
                check_tld_spoof(domain)['name'],
                answer,
                f'{domain} answer {answer} is incorrect',
            )

    def test_is_popular_domain(self):
        for domain, answer in (
            ('bingo.com.com', False),
            ('godaddy.gratis', False),
            ('istockphoto.hosting', False),
            ('admin.academy', False),
            ('undp.bet', False),
            ('flickr.tj', False),
            ('postgresql.global', False),
            ('amazon.brussels', False),
            ('nbcnews.berlin', False),
            ('azurewebsites.il', False),
            ('ihg.travel', False),
            ('elsevier.download', False),
            ('fb.guru', False),
            ('usatoday.ua', False),
            ('fr.marketing', False),
            ('cmu.agency', False),
            ('thestar.ph', False),
            ('arxiv.bf', False),
            ('stackexchange.ga', False),
            ('issuu.by', False),
            ('themeisle.pw', False),
            ('globalnews.game', False),
            ('ccc.sbi', False),
            ('uk.audio', False),
            ('dropcatch.bb', False),
            ('foxnews.gt', False),
            ('service.edu', False),
            ('colorado.int', False),
            ('teamviewer.int', False),
            ('indiana.org', False),
            ('gimp.us', False),
            ('metro.int', False),
            ('nasdaq.cn', False),
            ('chinadaily.uk', False),
            ('2.org', False),
            ('twitch.gle', False),
            ('readthedocs.edu', False),
            ('groups.ly', False),
            ('oracle.cn', False),
            ('blogspot.edu', False),
            ('thesun.eu', False),
            ('tumblr.us', False),
            ('bbb.be', False),
            ('mirror.cn', False),
            ('fda.net', False),
            ('tripadvisor.io', False),
            ('xbox.us', False),
            ('smithsonianmag.cn', False),
            ('openldap.gle', False),
            ('standard.gov', False),
            ('bing.com', True),
        ):
            self.assertEqual(
                is_popular_domain(domain),
                answer,
                f'{domain} answer {answer} is incorrect',
            )

    def test_check_unicode_blocks(self):
        for domain, answer in (
            ('раɣрɑǀ.com', FraudCheck.UnicodeBlock),
            ('ɡ၀၀ɡ𐌉ҽ.com', FraudCheck.UnicodeBlock),
            ('сha𐑈e.com', FraudCheck.UnicodeBlock),
            ('ԁısс၀⋁er.com', FraudCheck.UnicodeBlock),
            ('аmɑzဝn.com', FraudCheck.UnicodeBlock),
            ('whаtsаρρ.com', FraudCheck.UnicodeBlock),
            ('reԁԁᎥt.com', FraudCheck.UnicodeBlock),
            ('sр၀tify.com', FraudCheck.UnicodeBlock),
            ('nytımeѕ.com', FraudCheck.UnicodeBlock),
            ('ᖯinɡ.com', FraudCheck.UnicodeBlock),
            ('sհ၀ріfү.com', FraudCheck.UnicodeBlock),
            ('out׀၀၀k.com', FraudCheck.UnicodeBlock),
            ('teӀ℮ɡгam.com', FraudCheck.UnicodeBlock),
            ('ebаγ.com', FraudCheck.UnicodeBlock),
            ('twᎥtсհ.tv', FraudCheck.UnicodeBlock),
            ('ԁısс၀rԁ.dd', FraudCheck.UnicodeBlock),
            ('уe׀р.com', FraudCheck.UnicodeBlock),
            ('gooԁreаԁѕ.com', FraudCheck.UnicodeBlock),
            ('𐑈ɑmѕυng.com', FraudCheck.UnicodeBlock),
            ('reᏧhαt.com', FraudCheck.UnicodeBlock),
            ('econဝmɩ𐑈t.com', FraudCheck.UnicodeBlock),
            ('ınᏧеeԁ.com', FraudCheck.UnicodeBlock),
            ('асаdemᎥα.edu', FraudCheck.UnicodeBlock),
            ('сısс၀.com', FraudCheck.UnicodeBlock),
            ('f၀×news.com', FraudCheck.UnicodeBlock),
            ('newу၀rk℮r.com', FraudCheck.UnicodeBlock),
            ('уeاp.com', FraudCheck.UnicodeBlock),
            ('b၀၀kıng.com', FraudCheck.UnicodeBlock),
            ('руtհ၀n.org', FraudCheck.UnicodeBlock),
            ('аոԁr၀Ꭵᑯ.com', FraudCheck.UnicodeBlock),
            ('compassfoundation.io', FraudCheck.NoMatch),
        ):
            self.assertEqual(
                check_unicode_blocks(domain)['name'],
                answer,
                f'{domain} answer {answer} is incorrect',
            )

    def test_unicode_blocks(self):
        for domain, answer in (
            ('раɣрɑǀ.com', {'Cyrillic', 'IPA Extensions', 'Latin Extended-B', 'Basic Latin'}),
            ('сha𐑈e.com', {'Basic Latin', 'Cyrillic', 'Deseret'}),
            ('fасeb၀၀k.com', {'Basic Latin', 'Cyrillic', 'Myanmar'}),
            ('аmɑzဝn.com', {'Basic Latin', 'Cyrillic', 'IPA Extensions', 'Myanmar'}),
            ('compassfoundation.io', {'Basic Latin'}),
        ):
            self.assertEqual(
                unicode_blocks(domain),
                answer,
                f'{domain} answer {answer} is incorrect',
            )

    def test_ascii_attack(self):
        for domain, answer in (
            ('g00g1e.com', FraudCheck.Ascii),
            ('paypa1.com', FraudCheck.Ascii),
            ('disc0ver.com', FraudCheck.Ascii),
            ('amaz0n.com', FraudCheck.Ascii),
            ('sp0tify.com', FraudCheck.Ascii),
            ('outl00k.com', FraudCheck.Ascii),
            ('disc0rd.com', FraudCheck.Ascii),
            ('ye1p.com', FraudCheck.Ascii),
            ('g00dreads.com', FraudCheck.Ascii),
            ('ec0n0mist.com', FraudCheck.Ascii),
            ('cisc0.com', FraudCheck.Ascii),
            ('f0xnews.com', FraudCheck.Ascii),
            ('newy0rker.com', FraudCheck.Ascii),
            ('b00king.com', FraudCheck.Ascii),
            ('pyth0n.com', FraudCheck.Ascii),
            ('andr0id.com', FraudCheck.Ascii),
            ('c0mpassf0undation.com', FraudCheck.Ascii),
            ('0reilly.com', FraudCheck.Ascii),
            ('wa1mart.com', FraudCheck.Ascii),
            ('w3schoo1s.com', FraudCheck.Ascii),
            ('yah00.com', FraudCheck.Ascii),
            ('te1egram.org', FraudCheck.Ascii),
            ('z00m.com', FraudCheck.Ascii),
            ('c0deblocks.olivetree.software', FraudCheck.Ascii),
            ('h0medep0t.com', FraudCheck.Ascii),
            ('bing.com', FraudCheck.NoMatch),
        ):
            self.assertEqual(
                check_ascii_attack(domain)['name'],
                answer,
                f'{domain} answer {answer} is incorrect',
            )


if __name__ == '__main__':
    unittest.main()
