#!/usr/bin/env python3
import io
import os
from pathlib import Path
import re
from setuptools import setup, find_packages
from system_env import setup_portal_environment

PORTAL_ENV = setup_portal_environment()


def get_version():
    with open('dalmatian/__init__.py', 'r') as vf:
        line = vf.readline()
        match = re.match(r'__version__ = \'([\d\.]+)\'', line)

        if not match:
            raise ImportError("Can't read the version of dalmatian")

        version = match.group(1)
        return version


here = os.path.abspath(os.path.dirname(__file__))
with io.open(os.path.join(here, 'README.md'), encoding='utf-8') as f:
    long_description = '\n' + f.read()


def get_data_files():
    """
    Return all files under the root package, that are not in a
    package themselves.
    """
    files = [
        ('/etc/logrotate.d', ['config/logrotate/dalmatian']),
        # ('/var/log/dalmatian', ['config/logrotate/dalmatian.log']),
    ]

    for subscription in Path('subscriptions').iterdir():
        subscription_path = f'{PORTAL_ENV["DALMATIAN_DIR"]}/{subscription}'
        files.append((subscription_path, [f'{subscription}/settings.yml']),)

    return files


setup(
    name='dalmatian',
    version=get_version(),
    description="Dalmatian watchdog threat detector",
    long_description=long_description,
    long_description_content_type='text/markdown',
    author='Dave Burkholder',
    author_email='dave@thinkwelldesigns.com',
    packages=find_packages(exclude=["tests"]),
    license='SAP',
    zip_safe=True,
    url='https://code.compassfoundation.io/dave/dalmatian',
    data_files=get_data_files(),
    options={
        'bdist_rpm': {
            'post_install': 'post_install.sh',
            'pre_install': 'pre_install.sh',
        },
    },
    install_requires=[
        'invoke',
        'lchttp',
        'lcrsync',
        'lcrequests',
        'lcconfig',
        'netaddr',
        'unicodedataplus',
    ],
    classifiers=[
        'Environment :: Web Environment',
        'Intended Audience :: Developers',
        'Operating System :: OS Independent',
        'License :: OSI Approved :: MIT License',
        'Programming Language :: Python',
        'Topic :: Internet :: WWW/HTTP',
        'Topic :: Software Development :: Libraries :: Python Modules',
        'Programming Language :: Python :: 3.9',
        'Programming Language :: Python :: 3.11',
        'Programming Language :: Python :: Implementation :: CPython',
    ],
    entry_points={
        'console_scripts': ['dalmatian = dalmatian.main:program.run'],
    },
)
