import unicodedataplus
from enum import StrEnum
from typing import TypedDict

from .popular_domains import (
    read_website_domains,
    top_short_tld_pairs,
)

TOP_POPULAR_TLD_PAIRS = top_short_tld_pairs()
TOP_POPULAR_DOMAINS = read_website_domains()


class FraudCheck(StrEnum):
    NoMatch = 'No Fraud Detected'
    TldSpoof = 'Top Level Domain'
    UnicodeBlock = 'Many Unicode Blocks Found'
    Unicode = 'Look-alike Character'
    Punycode = 'Encoded Look-alike Character'
    Ascii = 'Character Substitute'


class Attack(TypedDict):
    name: FraudCheck
    targeted_domain: str


def perform_fraud_check(check_func, domain: str) -> Attack | None:
    """
    Call the fraud function check, and if matching,
    return the Attack result if the targeted domain
    is a popular domain.
    """
    if ((attack := check_func(domain))['name'] != FraudCheck.NoMatch
            and is_popular_domain(attack['targeted_domain'])):
        return attack


def check_unicode_homoglyph_attack(domain_name: str) -> Attack:
    """
    Check for domains that contain unicode characters
    that appear similar to standard ascii characters.
    EXAMPLE: bińg.com
    """
    if (converted_dom := domain_name.translate(DOMAIN_CHAR_TRANS)) != domain_name:
        return Attack(name=FraudCheck.Unicode, targeted_domain=converted_dom)

    return Attack(name=FraudCheck.NoMatch, targeted_domain=domain_name)


def check_tld_spoof(domain_name: str) -> Attack:
    """
    Check for domains that contain an additional tld at the end to spoof a valid domain.
    EXAMPLE: googlecom.org
    """  # noqa
    for tld1, tld2 in TOP_POPULAR_TLD_PAIRS:

        suffix = f'{tld1}.{tld2}'

        if (base_domain := domain_name.removesuffix(suffix)) != domain_name:
            base_domain = base_domain.rstrip('.')

            spoof1 = f'{base_domain}.{tld1}'
            if spoof1 in TOP_POPULAR_DOMAINS:
                return Attack(name=FraudCheck.TldSpoof, targeted_domain=spoof1)

            spoof2 = f'{base_domain}.{tld2}'
            if spoof2 in TOP_POPULAR_DOMAINS:
                return Attack(name=FraudCheck.TldSpoof, targeted_domain=spoof2)

    return Attack(name=FraudCheck.NoMatch, targeted_domain=domain_name)


def is_popular_domain(domain_name: str) -> bool:
    """
    Check that a domain is part of the list of popular domains.
    """
    return domain_name in TOP_POPULAR_DOMAINS


def check_punycode(domain_name: str) -> Attack:
    """
    Check that a domain encoded as punycode is contained in the list of valid domains.
    EXAMPLE: xn--big-qwa.com -> bińg.com -> bing.com
    """
    if domain_name.startswith('xn--'):
        if domain_name in TOP_POPULAR_DOMAINS:
            return Attack(name=FraudCheck.NoMatch, targeted_domain=domain_name)

        if domain_name.encode().decode('idna').translate(DOMAIN_CHAR_TRANS) in TOP_POPULAR_DOMAINS:
            return Attack(name=FraudCheck.Punycode, targeted_domain=domain_name)

    return Attack(name=FraudCheck.NoMatch, targeted_domain=domain_name)


def check_unicode_blocks(domain_name: str) -> Attack:
    """
    Check that a domain does not contain 3 or more unicode character blocks.
    EXAMPLE: раɣрɑǀ.com ->
    {'Basic Latin', 'Cyrillic', 'Greek and Coptic', 'Old Italic'}
    """  # noqa
    if len(unicode_blocks(domain_name)) >= 3:
        return Attack(name=FraudCheck.UnicodeBlock, targeted_domain=domain_name)

    return Attack(name=FraudCheck.NoMatch, targeted_domain=domain_name)


def unicode_blocks(name: str) -> set[str]:
    """
    Find the encoding of each character in string and return in a set.

    ['Basic Latin', 'Cyrillic', 'Greek and Coptic', 'Old Italic']
    """
    blocks = []
    for c in name:
        blocks.append(unicodedataplus.block(c))

    return set(blocks)


def check_ascii_attack(domain_name: str) -> Attack:
    """
    Check that a domain does not replace l and o with 1 and 0.
    EXAMPLE: g00g1e.com
    """
    if (normalized := domain_name.translate(ASCII_LOOKALIKES)) != domain_name:
        return Attack(name=FraudCheck.Ascii, targeted_domain=normalized)

    return Attack(name=FraudCheck.NoMatch, targeted_domain=domain_name)


ASCII_LOOKALIKES = str.maketrans({'1': 'l', '0': 'o'})
DOMAIN_CHAR_TRANS = str.maketrans({
    '𝟣': '1',
    '𝟭': '1',
    '𝟷': '1',
    'ｌ': '1',
    '１': '1',
    'ᒿ': '2',
    '２': '2',
    '𝟐': '2',
    '𝟤': '2',
    '𝟮': '2',
    '𝟸': '2',
    'З': '3',
    'Ӡ': '3',
    'Ⳍ': '3',
    'Ꝫ': '3',
    'Ɜ': '3',
    '３': '3',
    '𖼻': '3',
    '𝈆': '3',
    '𝟹': '3',
    '𝟥': '3',
    'Ꮞ': '4',
    '４': '4',
    '𝟒': '4',
    '𝟦': '4',
    '𝟰': '4',
    '𝟺': '4',
    'Ƽ': '5',
    '５': '5',
    '𑢻': '5',
    '𝟓': '5',
    '𝟧': '5',
    '𝟱': '5',
    '𝟻': '5',
    'б': '6',
    'Ꮾ': '6',
    'Ⳓ': '6',
    '６': '6',
    '𑣕': '6',
    '𝟔': '6',
    '𝟨': '6',
    '𝟲': '6',
    '𝟼': '6',
    '７': '7',
    '𐓒': '7',
    '𝟩': '7',
    '𝟳': '7',
    '𝟽': '7',
    '৪': '8',
    'ȣ': '8',
    'Ȣ': '8',
    '８': '8',
    '𝟪': '8',
    '𝟾': '8',
    '𝟴': '8',
    '𝟖': '8',
    '𐌚': '8',
    '୨': '9',
    'Ⳋ': '9',
    'Ꝯ': '9',
    '９': '9',
    '𝟗': '9',
    '𝟫': '9',
    '𝟵': '9',
    '𝟿': '9',
    '¡': 'i',
    'а': 'a',
    'ā': 'a',
    'ɑ': 'a',
    'α': 'a',
    'à': 'a',
    'á': 'a',
    'â': 'a',
    'ã': 'a',
    'ä': 'a',
    'ӓ': 'a',
    'å': 'a',
    'ȧ': 'a',
    'ǎ': 'a',
    'ă': 'a',
    'æ': 'ae',
    'ᖯ': 'b',
    'Ƅ': 'b',
    'Ь': 'b',
    'Ꮟ': 'b',
    'ᑲ': 'b',
    '𝐛': 'b',
    '𝚋': 'b',
    'ｂ': 'b',
    '𝖻': 'b',
    '𝗯': 'b',
    '𝖇': 'b',
    'ć': 'c',
    'ϲ': 'c',
    'с': 'c',
    'ᴄ': 'c',
    'ⅽ': 'c',
    'ⲥ': 'c',
    '𝖼': 'c',
    'ꮯ': 'c',
    'ｃ': 'c',
    '𝚌': 'c',
    '𝗰': 'c',
    '𐐽': 'c',
    'ç': 'c',
    'ꮷ': 'd',
    'ԁ': 'd',
    'ᑯ': 'd',
    'ⅾ': 'd',
    'đ': 'd',
    'ꓒ': 'd',
    'ｄ': 'd',
    '𝚍': 'd',
    '𝖽': 'd',
    'è': 'e',
    'ĕ': 'e',
    'ě': 'e',
    'é': 'e',
    'ê': 'e',
    'ę': 'e',
    'е': 'e',
    'ë': 'e',
    'ē': 'e',
    '℮': 'e',
    'ƒ': 'f',
    'ꬵ': 'f',
    'ｆ': 'f',
    '𝚏': 'f',
    'ǥ': 'g',
    'ğ': 'g',
    'ǧ': 'g',
    'ǵ': 'g',
    'ģ': 'g',
    'ɡ': 'g',
    'ց': 'g',
    'ᶃ': 'g',
    '𝚐': 'g',
    '𝗀': 'g',
    '𝗴': 'g',
    'հ': 'h',
    'һ': 'h',
    'Ꮒ': 'h',
    'ｈ': 'h',
    '𝚑': 'h',
    'ì': 'i',
    'ⅰ': 'i',
    'ｉ': 'i',
    'ꭵ': 'i',
    'í': 'i',
    'î': 'i',
    'ï': 'i',
    'ī': 'i',
    'і': 'i',
    'Ꭵ': 'i',
    'ı': 'i',
    'ј': 'j',
    'ϳ': 'j',
    '𝚓': 'j',
    'ｊ': 'j',
    'ｋ': 'k',
    'ǩ': 'k',
    '𝐤': 'k',
    '𝗄': 'k',
    '𝗸': 'k',
    '𝚔': 'k',
    'Ɩ': 'l',
    'ǀ': 'l',
    'ӏ': 'l',
    '𝚕': 'l',
    'ｍ': 'm',
    'ℳ': 'm',
    'ｎ': 'n',
    '𝚗': 'n',
    'ņ': 'n',
    'ɲ': 'n',
    '𝗻': 'n',
    '𝗇': 'n',
    'ñ': 'n',
    'ń': 'n',
    'ո': 'n',
    'ռ': 'n',
    'ŋ': 'n',
    'ò': 'o',
    'ဝ': 'o',
    'ó': 'o',
    'ô': 'o',
    'õ': 'o',
    'ö': 'o',
    'ø': 'o',
    'ŏ': 'o',
    'ǒ': 'o',
    'ō': 'o',
    '၀': 'o',
    'о': 'o',
    'ο': 'o',
    'օ': 'o',
    'ȯ': 'o',
    'ọ': 'o',
    'ỏ': 'o',
    'ð': 'o',
    'ơ': 'o',
    'ƿ': 'p',
    'þ': 'p',
    'ϸ': 'p',
    'р': 'p',
    'ρ': 'p',
    '𐓄': 'p',
    'Ϸ': 'p',
    'ɋ': 'q',
    'ᶐ': 'q',
    'զ': 'q',
    'ԛ': 'q',
    'գ': 'q',
    '𝐪': 'q',
    '𝗊': 'q',
    '𝗾': 'q',
    '𝙦': 'q',
    '𝚚': 'q',
    'ｑ': 'q',
    '৭': 'q',
    '੧': 'q',
    'ř': 'r',
    'г': 'r',
    '𝚛': 'r',
    'ꮁ': 'r',
    'ｒ': 'r',
    'ѕ': 's',
    'š': 's',
    '𐑈': 's',
    'ś': 's',
    'ʂ': 's',
    'ꜱ': 's',
    'ꮪ': 's',
    '𝚜': 's',
    'ṭ': 't',
    'ｔ': 't',
    '𝐭': 't',
    '𝚝': 't',
    'ţ': 't',
    'ƫ': 't',
    'ŧ': 't',
    'Ꮏ': 't',
    'ț': 't',
    'υ': 'u',
    'ù': 'u',
    'ս': 'u',
    'ú': 'u',
    'û': 'u',
    'ŭ': 'u',
    'ū': 'u',
    'ǔ': 'u',
    'ü': 'u',
    'ѵ': 'v',
    'ν': 'v',
    '∨': 'v',
    'ｖ': 'v',
    'ꮩ': 'v',
    'ⅴ': 'v',
    '𝝼': 'v',
    '⋁': 'v',
    'ԝ': 'w',
    'ᴡ': 'w',
    'ｗ': 'w',
    '𝚠': 'w',
    '𝗐': 'w',
    '𝘄': 'w',
    'х': 'x',
    'ҳ': 'x',
    '𝚡': 'x',
    'ⅹ': 'x',
    '⤫': 'x',
    '⤬': 'x',
    '⨯': 'x',
    'ｘ': 'x',
    '𝑥': 'x',
    '𝗑': 'x',
    '𝘅': 'x',
    '×': 'x',
    'γ': 'y',
    'ү': 'y',
    'у': 'y',
    'ɣ': 'y',
    'ý': 'y',
    'ÿ': 'y',
    'ᴢ': 'z',
    'ꮓ': 'z',
    'ｚ': 'z',
    '𝗓': 'z',
    '𝘇': 'z',
    '𝘻': 'z',
    '𝙯': 'z',
    '𝚣': 'z',
    'ʐ': 'z',
    'ź': 'z',
    'ž': 'z',
    'ż': 'z',
    '︰': ':',
})
