from aspen_crypto.encode import mask, unmask
from jaraco.classes import properties
from console_keyring.cryptfile.chachafile import ChaChaFileKeyring
from lcconfig import SymmetricKeysConfig
from lcconfig.settings import CONF_DIR
import secrets


class ConsoleKeyring(ChaChaFileKeyring):
    """
    Subclass and set several defaults for Consoles,
    mainly to automatically load keyring key from
    configuration file automatically.
    """

    _keyring_key: str | None

    filename = '.console_keyring.cfg'

    def get_password(self, service: str, username: str) -> str | None:
        """
        If no password can be read, ensure that the conf file is readable.
        """
        if password := super().get_password(service, username):
            return password
        with open(self.file_path) as c:
            c.readable()

    @properties.NonDataProperty
    def file_path(self) -> str:
        return f'{CONF_DIR}/{self.filename}'

    def _init_keyring(self) -> None:
        self.load_keyring_key_from_file()
        super()._init_keyring()

    def _unlock(self) -> None:
        self.load_keyring_key_from_file()
        super()._unlock()

    def load_keyring_key_from_file(self) -> None:
        """
        Check for existing key in encrypted config file
        and assign to keyring_key if found.
        """
        if self._keyring_key is not None:
            return

        symmetric_keys = SymmetricKeysConfig()
        cfg = symmetric_keys.as_typed_dict()
        try:
            self._keyring_key = unmask(cfg['chacha'])
            return
        except KeyError:
            pass

        # If file can't be read, raise permissions error
        with open(symmetric_keys.persist_file) as sk:
            sk.readable()

    @classmethod
    def generate_new_key(cls, key: str = '', force: bool = False) -> None:
        """
        Generate new encryption key, especially for new deployments.
        """
        symmetric_keys = SymmetricKeysConfig()
        cfg = symmetric_keys.as_typed_dict()

        if not force and cfg.get('chacha'):
            print('Keyring key already exists')
            return

        if key and len(key) != 32:
            raise ValueError('Key must be 32 bytes long!')

        cfg['chacha'] = mask(key or secrets.token_urlsafe(24))
        symmetric_keys.save_section(data=cfg)
