"""
Copyright (c) 2015 ORCAS, Inc
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of the author nor the names of other
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""

import pybase64
from cryptography.fernet import Fernet, MultiFernet
from typing import Any

from django.conf import settings
from django.core.exceptions import FieldError, ImproperlyConfigured
from django.db import models
from django.db.backends.postgresql.psycopg_any import is_psycopg3
from django.utils.encoding import force_bytes, force_str
from django.utils.functional import cached_property

from ..utils.hkdf import derive_fernet_key
from console_keyring.console_keyring import ConsoleKeyring

__all__ = (
    'EncryptedField',
    'EncryptedTextField',
    'EncryptedCharField',
    'EncryptedEmailField',
    'EncryptedIntegerField',
    'EncryptedDateField',
    'EncryptedDateTimeField',
    'EncryptedBinaryField',
)


class EncryptedField(models.Field):
    """A field that encrypts values using Fernet symmetric encryption."""

    _internal_type = 'BinaryField'

    def __init__(self, *args: Any, **kwargs: Any) -> None:
        if kwargs.get('primary_key'):
            raise ImproperlyConfigured(
                f"{self.__class__.__name__} does not support primary_key=True."
            )
        if kwargs.get('unique'):
            raise ImproperlyConfigured(f"{self.__class__.__name__} does not support unique=True.")
        if kwargs.get('db_index'):
            raise ImproperlyConfigured(f"{self.__class__.__name__} does not support db_index=True.")
        super(EncryptedField, self).__init__(*args, **kwargs)

    @cached_property
    def keys(self):
        """
        Load fernet keys from settings or from keyring.
        If loaded from keyring, assign to settings to optimize future lookups.
        """
        keys = getattr(settings, 'FERNET_KEYS', None)
        if not keys:
            kr = ConsoleKeyring()
            key = str(kr.get_password('console', 'fernet') or '').encode()
            if not key:
                key = pybase64.urlsafe_b64encode(settings.SECRET_KEY[:32].encode())
            settings.FERNET_KEYS = keys = [key]
        return keys

    @cached_property
    def fernet_keys(self):
        if getattr(settings, 'FERNET_USE_HKDF', True):
            return [derive_fernet_key(k) for k in self.keys]
        return self.keys

    @cached_property
    def fernet(self):
        if len(self.fernet_keys) == 1:
            return Fernet(self.fernet_keys[0])
        return MultiFernet([Fernet(k) for k in self.fernet_keys])

    def get_internal_type(self):
        return self._internal_type

    def get_db_prep_save(self, value, connection):
        value = super(EncryptedField, self).get_db_prep_save(value, connection)
        if value is not None:
            if is_psycopg3 and hasattr(value, 'obj'):
                value = value.obj
            retval = self.fernet.encrypt(force_bytes(value))
            return connection.Database.Binary(retval)

    def from_db_value(self, value, expression, connection, *args):
        if value is not None:
            value = bytes(value)
            return self.to_python(force_str(self.fernet.decrypt(value)))

    @cached_property
    def validators(self):
        # Temporarily pretend to be whatever type of field we're masquerading
        # as, for purposes of constructing validators (needed for
        # IntegerField and subclasses).
        self.__dict__['_internal_type'] = super(EncryptedField, self).get_internal_type()
        try:
            return super(EncryptedField, self).validators
        finally:
            del self.__dict__['_internal_type']


def get_prep_lookup(self):
    """Raise errors for unsupported lookups"""
    raise FieldError(
        f"{self.lhs.field.__class__.__name__} {self.lookup_name!r} does not support lookups"
    )


# Register all field lookups (except 'isnull') to our handler
for name, lookup in models.Field.class_lookups.items():
    # Dynamically create classes that inherit from the right lookups
    if name != 'isnull':
        lookup_class = type(
            'EncryptedField' + name, (lookup,), {'get_prep_lookup': get_prep_lookup}
        )
        EncryptedField.register_lookup(lookup_class)


class EncryptedTextField(EncryptedField, models.TextField):
    pass


class EncryptedCharField(EncryptedField, models.CharField):
    pass


class EncryptedEmailField(EncryptedField, models.EmailField):
    pass


class EncryptedIntegerField(EncryptedField, models.IntegerField):
    pass


class EncryptedDateField(EncryptedField, models.DateField):
    pass


class EncryptedDateTimeField(EncryptedField, models.DateTimeField):
    pass


class EncryptedBinaryField(EncryptedField, models.BinaryField):
    def from_db_value(self, value, expression, connection, *args):
        if value is not None:
            decrypted_value = self.to_python(self.fernet.decrypt(bytes(value)))
            if decrypted_value.endswith(b'::bytea'):
                decrypted_value = decrypted_value[:-7]
            return decrypted_value.replace(b'\\012', b'\n').strip(b'"\'').strip()
