from uuid import UUID
from typing import Optional, TYPE_CHECKING
from django.contrib.auth import get_user_model
from django.db import DatabaseError

if TYPE_CHECKING:
    from sandbox.latchstring.models import User as UserType

User = get_user_model()


def confirm_api_user(
    name: str,
    cid: UUID,
    user: Optional['UserType'] = None,
    is_superuser: bool = True,
) -> 'UserType':
    """
    Make sure an API user has been created, with the needed token.
    Used for auto-creating API user on Consoles, Aspen, etc., and
    having a predictable Canonical ID for the API User is useful.
    """
    if user:
        if str(user.cid) != str(cid):
            user.cid = cid
            try:
                user.save(update_fields=['cid'])
            except DatabaseError:
                User.objects.filter(cid=cid).delete()
                user.save(update_fields=['cid'])
        return user

    if is_superuser:
        privilege_flags = {
            'is_superuser': True,
            'is_staff': True,
        }
    else:
        privilege_flags = {}

    api_user = User.objects.create(
        first_name=name.title(),
        last_name='API',
        username=f'{name}_api',
        email=f'support+{name}_api@compassfoundation.io',
        is_api=True,
        cid=cid,
        **privilege_flags,
    )

    return api_user
