"""
Base DRF viewsets for Console API so that
customizations can consistently and quickly be
deployed throughout the entire application.
"""

import logging

from rest_framework import mixins
from rest_framework.generics import GenericAPIView
from rest_framework.viewsets import (
    ViewSetMixin,
)
from console_base.api.mixins import (
    AllowPUTAsCreateMixin,
    DataTablesMixin,
    SelectFieldMixin,
    TenantQueryMixin,
    PermissionRequiredMixin,
)

from .views import LCAPIView

logger = logging.getLogger(__name__)


# ---------------------------------------------------------------------------
class LCViewSetMixin(ViewSetMixin):
    pass


# ---------------------------------------------------------------------------
class LCViewSet(LCViewSetMixin, LCAPIView):
    """
    Base ViewSet for the Console
    """

    pass


# ---------------------------------------------------------------------------
class LCGenericViewSet(LCViewSetMixin, GenericAPIView):
    """
    Base GenericViewSet for the Console
    """

    lookup_field = 'cid'


# ---------------------------------------------------------------------------
class LCReadOnlyModelViewSet(
    PermissionRequiredMixin,
    TenantQueryMixin,
    DataTablesMixin,
    SelectFieldMixin,
    mixins.RetrieveModelMixin,
    mixins.ListModelMixin,
    LCGenericViewSet,
):
    """
    A viewset that provides default `list()` and `retrieve()` actions.
    """

    pass


# ---------------------------------------------------------------------------
class LCModelViewSet(
    PermissionRequiredMixin,
    TenantQueryMixin,
    DataTablesMixin,
    SelectFieldMixin,
    AllowPUTAsCreateMixin,
    mixins.CreateModelMixin,
    mixins.RetrieveModelMixin,
    mixins.UpdateModelMixin,
    mixins.DestroyModelMixin,
    mixins.ListModelMixin,
    LCGenericViewSet,
):
    """
    Base ModelViewSet for the Console
    """

    pass


__all__ = (
    'LCViewSetMixin',
    'LCViewSet',
    'LCGenericViewSet',
    'LCModelViewSet',
    'LCReadOnlyModelViewSet',
)
