import json
import logging
from vanilla import DeleteView, DetailView, TemplateView, UpdateView
from webauthn import (
    generate_authentication_options,
    generate_registration_options,
    options_to_json,
    verify_authentication_response,
    verify_registration_response,
    base64url_to_bytes,
)
from webauthn.helpers.exceptions import InvalidRegistrationResponse, InvalidAuthenticationResponse
from webauthn.helpers.structs import (
    AuthenticationCredential,
    PublicKeyCredentialDescriptor,
    RegistrationCredential,
)
from django.conf import settings
from django.contrib import messages
from django.contrib.auth import login
from django.http import HttpResponse, HttpResponseRedirect
from django.shortcuts import get_object_or_404, redirect, render
from django.template.response import TemplateResponse
from django.urls import reverse
from django.utils.translation import gettext_lazy as _

from console_base.views import (
    LayoutSettings,
    LCDeleteViewBase,
    LCDetailViewBase,
    LCUpdateViewBase,
    LCTemplateViewBase,
    LCOpenTemplateViewBase,
)
from .forms import CredentialForm
from .models import (
    User,
    Authentication,
    Credential,
    Registration,
)

logger = logging.getLogger(__name__)


# ----------------------------------------------------------------------
class WebAuthnMixin:
    """
    Mixin class to determine whether a WebAuthn authentication
    is required before authenticating the user session.
    """

    def form_valid(self, form):
        """
        Check if the user uses webauthn
        """
        user: User = form.get_user()
        if not user:
            return super().form_valid(form)

        auth_backend = getattr(user, 'backend', None)

        if (auth_backend in settings.BACKENDS_REQUIRING_WEBAUTHN
                and user.webauthn.active().count() > 0):
            self.request.session[settings.USER_PK_KEY] = str(user.pk)
            if auth_backend:
                self.request.session[settings.AUTH_BACKEND_KEY] = auth_backend
            url = f'{reverse("console_webauthn:login")}?next={self.request.GET.get("next")}'
            return HttpResponseRedirect(url)

        return super().form_valid(form)


# -------------------------------------------------------------------------
class WebAuthnRegister(LayoutSettings, LCTemplateViewBase, TemplateView):
    permission_required = settings.NO_PERM_CHECK
    template_name = f'webauthn_register.{settings.TEMPLATE_EXTENSION}'

    def name(self):
        return _('Register WebAuthn Credential')

    def get(self, request, *args, **kwargs):

        public_credential_creation_options = generate_registration_options(
            rp_id=request.get_host(),
            rp_name=settings.CONSOLE_NAME,
            user_id=str(request.user.pk),
            user_name=request.user.username,
        )

        webauthn_registration, _ = Registration.objects.get_or_create(user=request.user)

        # The options are encoded in bytes - to save them correctly
        # I use the json strings
        options_dict = json.loads(options_to_json(public_credential_creation_options))
        webauthn_registration.challenge = options_dict["challenge"]
        webauthn_registration.save()

        return render(
            request,
            self.template_name,
            context={
                "view": self,
                "public_credential_creation_options": options_to_json(
                    public_credential_creation_options
                )
            },
        )

    def post(self, request, *args, **kwargs):

        webauthn_registration = Registration.objects.get(user=request.user)
        registration_credentials = RegistrationCredential.parse_raw(request.body)

        try:
            authentication_verification = verify_registration_response(
                credential=registration_credentials,
                expected_challenge=base64url_to_bytes(webauthn_registration.challenge),
                expected_origin=f"https://{request.get_host()}", # dont forget the ports
                expected_rp_id=request.get_host(),
            )

            auth_json = json.loads(authentication_verification.json())
            Credential.objects.create(
                user=request.user,
                credential_public_key=auth_json.get("credential_public_key"),
                credential_id=auth_json.get("credential_id"),
            )

            return HttpResponse(status=201)
        except InvalidRegistrationResponse as error:
            messages.error(request, f"Something went wrong: {error}")
            return redirect("console_webauthn:register")


# -------------------------------------------------------------------------
class WebAuthnLogin(LayoutSettings, LCOpenTemplateViewBase, TemplateView):
    template_name = f'webauthn_login.{settings.TEMPLATE_EXTENSION}'

    def name(self):
        return _('Login Via WebAuthn Credential')

    def get(self, request, *args, **kwargs):

        user = get_object_or_404(User, pk=request.session.get(settings.USER_PK_KEY))
        authenticators = user.webauthn.all()
        webauthn_authentication, _ = Authentication.objects.get_or_create(user=user)

        allowed_credentials = [
            PublicKeyCredentialDescriptor(id=base64url_to_bytes(credentials.credential_id))
            for credentials in authenticators
        ]
        authentication_options = generate_authentication_options(
            rp_id=request.get_host(),
            allow_credentials=allowed_credentials,
        )
        json_option = json.loads(options_to_json(authentication_options))
        webauthn_authentication.challenge = json_option.get("challenge")
        webauthn_authentication.save()

        return render(
            request,
            self.template_name,
            context={
                "authentication_options": options_to_json(authentication_options),
                "next": request.GET.get("next") or reverse('latchstring:user_profile'),
                "view": self,
            },
        )

    def post(self, request, *args, **kwargs):
        auth_backend = request.session.get(settings.AUTH_BACKEND_KEY)
        user = get_object_or_404(User, pk=request.session.get(settings.USER_PK_KEY))

        webauthn_authentication = Authentication.objects.get(user=user)
        authentication_cred = AuthenticationCredential.parse_raw(request.body)

        try:
            # only lookup active credentials, so that soft deletes can remove the key from use
            webauthn_cred = Credential.objects.active().get(credential_id=authentication_cred.id)
        except Credential.DoesNotExist:
            self.clear_webauthn_session_keys(request.session)
            return HttpResponse(b'No active credential found!', status=403)

        if webauthn_cred.user != user:
            logger.error(
                'WebAuth credentials %s user does not match session user %s' %
                (webauthn_cred.user, user)
            )
            self.clear_webauthn_session_keys(request.session)
            return HttpResponse(b'Credential user does not match session user!', status=403)

        try:
            authentication_verification = verify_authentication_response(
                credential=authentication_cred,
                expected_challenge=base64url_to_bytes(webauthn_authentication.challenge),
                expected_rp_id=request.get_host(),
                expected_origin=f"https://{request.get_host()}",
                credential_public_key=base64url_to_bytes(webauthn_cred.credential_public_key),
                credential_current_sign_count=webauthn_cred.current_sign_count,
            )
        except InvalidAuthenticationResponse as error:
            logger.exception(error)
            return HttpResponse(status=403)

        webauthn_cred.current_sign_count = authentication_verification.new_sign_count
        webauthn_cred.save()
        self.clear_webauthn_session_keys(request.session)

        try:
            login(request, user, backend=auth_backend)
        except Exception as e:
            logger.exception(e)

        return HttpResponse(status=200)

    def clear_webauthn_session_keys(self, session):
        """
        Clear the session variables that the webauthn flow adds.
        """
        for key in (settings.AUTH_BACKEND_KEY, settings.USER_PK_KEY):
            try:
                del session[key]
            except KeyError:
                pass


# ----------------------------------------------------------------------
class CredentialUpdate(LayoutSettings, LCUpdateViewBase, UpdateView):
    permission_required = 'webauthn.change_credential'
    template_name = f'record_create.{settings.TEMPLATE_EXTENSION}'
    model = Credential
    form_class = CredentialForm

    def send_optimized_success_response(self) -> bool:
        return True

    def get_unpoly_target(self) -> str:
        return f'#{self.get_object().row_pk}'

    def optimized_success_response(self):

        return TemplateResponse(
            self.request,
            f'single_row_table.{settings.TEMPLATE_EXTENSION}',
            {
                'body_id': 'webauthn_credentials_body',
                'table_row_template': f'snippets/webauthn_credential_row.{settings.TEMPLATE_EXTENSION}',
                'credential': self.object,
                'view': self,
            },
        )


# ----------------------------------------------------------------------
class CredentialDetail(LayoutSettings, LCDetailViewBase, DetailView):
    permission_required = 'webauthn.view_credential'
    template_name = f'detail.{settings.TEMPLATE_EXTENSION}'
    model = Credential


# ----------------------------------------------------------------------
class CredentialDelete(LayoutSettings, LCDeleteViewBase, DeleteView):
    permission_required = 'webauthn.delete_credential'
    template_name = f'detail.{settings.TEMPLATE_EXTENSION}'
    unpoly_modal_template = f'confirmation_delete.{settings.TEMPLATE_EXTENSION}'
    model = Credential
