from typing import TYPE_CHECKING

from django.apps import apps
from django.utils import timezone

from concordia.events.trail import ConcordEventsTrail
from console_deps.datetime_truncate import truncate_day
from console_base.views import LCDetailViewBase

if TYPE_CHECKING:
    ActivityStreamBase = LCDetailViewBase
else:
    ActivityStreamBase = object


# ----------------------------------------------------------------------
class ActivityStreamViewerMixin(ActivityStreamBase):  # type: ignore

    def update_last_viewed(self):
        self.request.user.last_viewed_events = timezone.now()
        self.request.user.save(update_fields=['last_viewed_events'])

    def get_context_data(self, **kwargs):

        user = self.request.user
        events = ConcordEventsTrail(user.last_viewed_events)
        if not events.unviewed_events_exist():
            # use truncate_date to retain the timezone
            events.date = truncate_day(timezone.localtime())

        self.update_last_viewed()

        return super().get_context_data(
            events_since=events.date,
            event_stream=events.unviewed_events(),
            **kwargs,
        )


# ----------------------------------------------------------------------
class ActivityStreamRecordViewerMixin(ActivityStreamBase):  # type: ignore
    """
    Get activity stream for entire life cycle of this record
    """

    def get_record(self):
        Model = apps.get_model(app_label=self.kwargs['app'], model_name=self.kwargs['model'])
        return Model.objects.filter(cid=self.kwargs['cid']).first()  # noqa

    def get_context_data(self, **kwargs):

        record = self.get_record()
        if record:
            events = ConcordEventsTrail(record.created).record_events(target=record)
        else:
            events = []

        return super().get_context_data(
            events_since=record.created if record else timezone.localtime(),
            event_stream=events,
            **kwargs,
        )
