from collections import defaultdict
import logging

from django.apps import apps
from django.conf import settings

from ..models import ConcordEvents

logger = logging.getLogger(__name__)


# ---------------------------------------------------------------------------
class ConcordEventsTrail:
    """
    Handle retrieval and display of Record Events in an Audit Trail view.
    """

    def __init__(self, date=None):
        self.date = date
        self._event_models = {}

    def event_models(self) -> dict:
        """
        Return all models in the App that are Record Event tracking models.
        """
        if not self._event_models:
            event_models = defaultdict(list)
            for app, models in apps.all_models.items():
                if app not in settings.AUDITABLE_APPS:
                    continue
                for name, model_class in models.items():
                    if name.endswith('event'):
                        event_models[app].append(model_class)
            self._event_models = dict(event_models)
        return self._event_models

    def unviewed_events_exist(self) -> bool:
        """
        Check to see if any events have occurred since the last date specified.
        """
        for app, models in self.event_models().items():
            for EventModel in models:
                if EventModel.objects.filter(pgh_created_at__gte=self.date).exists():
                    return True
        return False

    def unviewed_events(self) -> list:
        """
        Return all Events that occurred after the specified date for display to the User.
        """
        all_events = []

        for app, Models in self.event_models().items():

            for EventModel in Models:
                event_models = EventModel.objects.filter(
                    pgh_created_at__gte=self.date,
                ).select_related('pgh_obj').distinct('pgh_obj')

                for event_model in event_models:
                    console_record = event_model.pgh_obj
                    console_record_url = console_record.get_absolute_url()

                    events = ConcordEvents.objects.tracks(console_record).filter(
                        pgh_created_at__gte=self.date,
                    ).select_related(
                        'user',
                    ).order_by('-pgh_created_at')

                    for event in events:
                        all_events.append({
                            'model_name': console_record.__class__.__name__,
                            'css_icon': console_record.css_icon,
                            'record': event,
                            'record_url': console_record_url,
                            'record_name': str(console_record),
                        })

        all_events.sort(key=lambda e: e['record'].pgh_created_at, reverse=True)

        return all_events

    def record_events(self, target) -> list:
        """
        Return all Events for the specified target record.
        """
        all_events = []
        url = target.get_absolute_url()

        events = ConcordEvents.objects.tracks(target).filter(
            pgh_created_at__gte=self.date,
        ).select_related(
            'user',
        ).order_by('-pgh_created_at')

        for event in events:
            all_events.append({
                'model_name': target.__class__.__name__,
                'css_icon': target.css_icon,
                'record': event,
                'record_url': url,
                'record_name': str(target),
            })

        return all_events
