# Generate self-signed keys for Clarion Server, so
# the nginx config file is valid, and the clarion
# domains are live for LetsEncrypt certbot.
import click
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives import serialization
from cryptography.hazmat.primitives.asymmetric import rsa
from cryptography import x509
from cryptography.x509.oid import NameOID
from cryptography.hazmat.primitives import hashes
from datetime import datetime, timedelta
from pathlib import Path

PRIVATE_KEY_PATH = '/etc/ssl/clarion_self_signed.key'
CLARION_PEM_PATH = '/etc/ssl/clarion_self_signed.pem'


# ---------------------------------------------------------------------------
def save_private_key(key) -> rsa.RSAPrivateKey:
    """
    Save a Private key to the specified path
    """
    with open(PRIVATE_KEY_PATH, 'wb') as f:
        f.write(
            key.private_bytes(
                encoding=serialization.Encoding.PEM,
                format=serialization.PrivateFormat.TraditionalOpenSSL,
                encryption_algorithm=serialization.NoEncryption(),
            )
        )

    return key


# ---------------------------------------------------------------------------
def save_x509_pem(pem: x509.Certificate) -> x509.Certificate:
    """
    Save an x509 Certificate to the specified path
    """
    with open(CLARION_PEM_PATH, 'wb') as f:
        f.write(pem.public_bytes(serialization.Encoding.PEM))

    return pem


# ---------------------------------------------------------------------------
def generate_key():
    """
    Generate a Private key for the Clarion Console
    and save to the specified path
    """

    key = rsa.generate_private_key(
        public_exponent=65537,
        key_size=2048,
        backend=default_backend(),
    )

    save_private_key(key)

    return key


# ---------------------------------------------------------------------------
def generate_certificate(key, hostname, dnsnames=()):
    """
    Generate Certificate Signing request
    """
    altnames = []
    for dns in dnsnames:
        altnames.append(x509.DNSName(dns))

    name = x509.Name([
        x509.NameAttribute(NameOID.COUNTRY_NAME, 'US'),
        x509.NameAttribute(NameOID.STATE_OR_PROVINCE_NAME, 'PA'),
        x509.NameAttribute(NameOID.LOCALITY_NAME, 'Clarion'),
        x509.NameAttribute(NameOID.ORGANIZATION_NAME, 'Clarion'),
        x509.NameAttribute(NameOID.COMMON_NAME, hostname),
        x509.NameAttribute(NameOID.EMAIL_ADDRESS, 'support@clarion.im'),
    ])

    certificate = x509.CertificateBuilder(
        serial_number=x509.random_serial_number(),
        not_valid_before=datetime.today() - timedelta(days=1),
        not_valid_after=datetime.today() + timedelta(days=365),
    ).subject_name(
        name,
    ).issuer_name(
        name,
    ).public_key(key.public_key()).add_extension(
        x509.SubjectAlternativeName(altnames),
        critical=False,
    ).add_extension(
        x509.BasicConstraints(ca=False, path_length=None),
        critical=True,
    ).add_extension(
        x509.ExtendedKeyUsage([x509.OID_SERVER_AUTH, x509.OID_CLIENT_AUTH]),
        critical=True,
    ).sign(key, hashes.SHA256(), default_backend())

    return save_x509_pem(certificate)


def generate_self_signed_ssl(homeserver_server_name, push_server_name, web_client_server_name, directory_server_name):
    """
    Generate self-signed cert for Clarion homeserver, so the
    nginx config file is valid, and Certbot can start.
    """
    if Path(CLARION_PEM_PATH).exists():
        click.echo(f'{CLARION_PEM_PATH} already exists')
        return

    private_key = generate_key()
    generate_certificate(
        private_key,
        hostname=homeserver_server_name,
        dnsnames=(web_client_server_name, directory_server_name),
    )
