import click
from pathlib import Path
import psycopg2
from psycopg2 import sql
from psycopg2.extensions import ISOLATION_LEVEL_AUTOCOMMIT
from subprocess import run, DEVNULL
import sys

from claradm import cli
from claradm.settings import CLARION_HOMESERVER_DIR, CLARION_BASE_WEBSERVER_DIR
from claradm.ini_cfg import DirectoryGeneral
from claradm.tls import generate_self_signed_ssl
from claradm.yaml_cfg import (
    ClaradmYAML,
    EmailConfig,
    FederationConfig,
    HomeServerConfig,
    MaintenanceConfig,
    ModulesConfig,
    OidcConfig,
    PasswordJwtConfig,
    PushConfig,
    RegistrationConfig,
    RoomsConfig,
    ThreePidConfig,
)

@cli.root.group(chain=True)
def setup():
    """ Perform Clarion setup and configuration tasks.
    """


@setup.command(name="display")
@click.option(
    "--name",
    default='homeserver',
    help="""Display contents of specified config file  [default: homeserver].""")
@click.pass_context
def display(ctx, name):
    """ Display the contents of the specified configuration file.
    """
    name = f'{name}.yaml' if not name.endswith('.yaml') else name

    click.echo(f'{CLARION_HOMESERVER_DIR}/conf.d/{name}')
    file_name = Path(f'{CLARION_HOMESERVER_DIR}/conf.d/{name}')

    if not file_name.exists():
        raise SystemExit(f'{name} config file does not exist')

    yaml = ClaradmYAML()
    data = yaml.load(file_name)

    click.echo(yaml.dump(data, sys.stdout))


@setup.command()
@click.pass_context
def database(ctx):
    """ Create Clarion database.
    """
    con = psycopg2.connect(user='postgres', host='/tmp')
    con.set_isolation_level(ISOLATION_LEVEL_AUTOCOMMIT)
    cur = con.cursor()
    db_role_name = 'clarion'

    # Ensure the clarion role exists.
    cur.execute("SELECT COUNT(*) FROM pg_catalog.pg_roles WHERE rolname = %s", [db_role_name])
    role_count, = cur.fetchone()

    if role_count:
        click.echo(f'Database role {db_role_name!r} already exists.')
    else:
        query = sql.SQL("CREATE ROLE {0} WITH login superuser").format(
            sql.Identifier(db_role_name),
        )
        cur = con.cursor()
        cur.execute(query.as_string(con))
        cur.execute("COMMIT")
        click.echo(f'Created {db_role_name!r} database role.')

    # Ensure the clarion database exists.
    cur.execute("SELECT COUNT(*) FROM pg_database WHERE datname = %s", [db_role_name])
    database_count, = cur.fetchone()
    if database_count:
        click.echo(f'Database {db_role_name!r} already exists.')
    else:
        query = sql.SQL("CREATE DATABASE {0} WITH ENCODING='UTF8' OWNER={1} LOCALE='C' TEMPLATE='template0'").format(
            sql.Identifier(db_role_name),
            sql.Identifier(db_role_name),
        )
        cur = con.cursor()
        cur.execute(query.as_string(con))
        cur.execute("COMMIT")
        click.echo(f'Created {db_role_name!r} database.')


@setup.command()
@click.pass_context
def webserver(ctx):
    """ Create Clarion webserver config file if it doesn't exist.
    """
    webserver_conf = '/etc/nginx/conf.d/clarion.conf'
    if Path(webserver_conf).exists():
        click.echo(f'Webserver conf file already exists at {webserver_conf!r}')
        return

    hs_data = HomeServerConfig().load_config()
    directory_data = DirectoryGeneral().load()
    config_data = {
        'homeserver_server_name': hs_data['server_name'],
        'push_server_name': f'{hs_data["subdomain"]}-push.{hs_data["clarion_service"]}',
        'web_client_server_name': f'{hs_data["subdomain"]}-web.{hs_data["clarion_service"]}',
        'directory_server_name': directory_data.get_value('server.name'),
    }

    with open(f'{CLARION_BASE_WEBSERVER_DIR}/clarion.conf') as bf:
        base_config = bf.read()

    with open(webserver_conf, 'w') as wf:
        wf.write(base_config % config_data)

    generate_self_signed_ssl(**config_data)
    run(['service', 'nginx', 'restart'], stdout=DEVNULL)


@setup.command()
@click.pass_context
def email(ctx):
    """ Configure email settings.
    """
    yaml = EmailConfig()
    yaml.collect_data()


@setup.command()
@click.pass_context
def federation(ctx):
    """ Configure federation settings.
    """
    yaml = FederationConfig()
    yaml.collect_data()


@setup.command()
@click.pass_context
def homeserver(ctx):
    """ Configure Clarion homeserver settings. The following settings are defined here:

    * `server_name` - The public-facing domain of the server. The server_name name will
    appear at the end of usernames and room addresses created on your server.
    For example if the server_name was tom.clarion.im, usernames on your server would
    be in the format @tom.clarion.im

    * `report_stats` - Whether or not to report homeserver usage statistics.

    * `registration_shared_secret` - Allows registration of standard or admin accounts by
    anyone who has the shared secret, even if enable_registration is not set.

    * `macaroon_secret_key` - A secret which is used to sign:
        - access token for guest users
        - short-term login token used during SSO logins (OIDC or SAML2)
        - token used for unsubscribing from email notifications

    * `form_secret` - A secret which is used to calculate HMACs for form values, to stop
    falsification of values. Must be specified for the User Consent forms to work.
    """
    yaml = HomeServerConfig()
    yaml.collect_data()


@setup.command()
@click.pass_context
def maintenance(ctx):
    """ Background updates are database updates that are run in the background in batches.
    The duration, minimum batch size, default batch size, whether to sleep between batches
    and if so, how long to sleep can all be configured. This is helpful to speed up or slow
    down the updates. This setting has the following sub-options:
    """
    yaml = MaintenanceConfig()
    yaml.collect_data()


@setup.command()
@click.pass_context
def modules(ctx):
    """ Configure modules settings.
    """
    click.echo('Module support not implemented')
    # yaml = ModulesConfig()
    # yaml.collect_data()


@setup.command()
@click.pass_context
def sso(ctx):
    """ Additional settings to use with single-sign on systems such as Clavis,
     OpenID Connect, SAML2 and CAS.

    Server admins can configure custom templates for pages related to SSO.
    See here for more information.
    """
    yaml = OidcConfig()
    yaml.collect_data()


@setup.command()
@click.pass_context
def password_jwt(ctx):
    """ JSON web token integration. The following settings can be used to make Clarion use
     JSON web tokens for authentication, instead of its internal password database.

    Each JSON Web Token needs to contain a "sub" (subject) claim, which is used as
    the localpart of the mxid.

    Additionally, the expiration time ("exp"), not before time ("nbf"), and issued at
    ("iat") claims are validated if present.

    Note that this is a non-standard login type and client support is expected to be non-existent.
    """
    yaml = PasswordJwtConfig()
    yaml.collect_data()


@setup.command()
@click.pass_context
def push(ctx):
    """ Configuration settings related to push notifications. . The following settings are defined:

    * `include_content` - Clients requesting push notifications can either have the body of the
    message sent in the notification poke along with other details like the sender, or just
    the event ID and room ID. For modern android devices the notification content will still
    appear because it is loaded by the app. iPhone, however will send a notification saying
    only that a message arrived and who it came from. Defaults to true. Set to false to only
    include the event ID and room ID in push notification payloads.

    * `group_unread_count_by_room` - hen a push notification is received, an unread count is also
    sent. This number can either be calculated as the number of unread messages for the user,
    or the number of rooms the user has unread messages in. Defaults to true, meaning push
    clients will see the number of rooms with unread messages in them. Set to false to instead
    send the number of unread messages.
    """
    yaml = PushConfig()
    yaml.collect_data()


@setup.command()
@click.pass_context
def registration(ctx):
    """ Configure Account Registration settings. The following settings are defined:

    * `enable_registration` - Enable registration for new users. Defaults to false.

    * `enable_registration_without_verification` - Enable registration without email or captcha
    verification. Note: this option is not recommended, as registration without verification
    is a known vector for spam and abuse. Defaults to false. Has no effect unless
    `enable_registration` is also enabled.

    * `session_lifetime` - Time that a user's session remains valid for, after they log in.

    * `refreshable_access_token_lifetime` - Time that an access token remains valid for, if the
    session is using refresh tokens.

    * `refresh_token_lifetime` - Time that a refresh token remains valid for (provided that it
    is not exchanged for another one first). This option can be used to automatically log-out
    inactive sessions.

    * `nonrefreshable_access_token_lifetime` - Time that an access token remains valid for,
    if the session is NOT using refresh tokens. Please note that not all clients support
    refresh tokens, so setting this to a short value may be inconvenient for some users who
    will then be logged out frequently.

    * `allow_guest_access` - Allows users to register as guests without a password/email/etc,
    and participate in rooms hosted on this server which have been made accessible to
    anonymous users. Defaults to false.

    * `enable_set_displayname`
    """
    yaml = RegistrationConfig()
    yaml.collect_data()


@setup.command()
@click.pass_context
def rooms(ctx):
    """ Config options relating to rooms. The following settings are defined:

    * `enabled` - Defines whether users can search the user directory. If false then empty
    responses are returned to all queries.

    * `search_all_users` - Defines whether to search all users visible to your HS when searching
    the user directory. If false, search results will only contain users visible in public
    rooms and users sharing a room with the requester. Defaults to false.

    * `enable_room_list_search` - Set to false to disable searching the public room list. When
    disabled blocks searching local and remote room lists for local and remote users by always
    returning an empty list for all queries. Defaults to true.
    """
    yaml = RoomsConfig()
    yaml.collect_data()


@setup.command()
@click.pass_context
def threepid(ctx):
    """ Config options relating Third Party IDs.
    """
    yaml = ThreePidConfig()
    yaml.collect_data()


@setup.command()
@click.pass_context
def enable(ctx):
    """ Enable Clarion server systemd services.
    """
    run(['systemctl', 'daemon-reload'], stdout=DEVNULL)
    run(['systemctl', 'enable', 'clarion-worker@federation.service'], stdout=DEVNULL)
    run(['systemctl', 'enable', 'clarion-worker@client.service'], stdout=DEVNULL)
    run(['systemctl', 'enable', 'clarion-directory.service'], stdout=DEVNULL)
    run(['systemctl', 'enable', 'clarion-server.service'], stdout=DEVNULL)
    run(['systemctl', 'enable', 'clarion-server.target'], stdout=DEVNULL)


def confirm_signing_keys():
    """
    Ensure that signing keys are present for Synapse & Sydent.
    """
    hs = HomeServerConfig().load_config()
    if not Path(hs['signing_key_path']).exists():
        run([
            'python3',
            '-m',
            'synapse.app.homeserver',
            '--config-path',
            '/etc/clarion/homeserver/conf.d',
            '--generate-missing-configs',
        ])


@setup.command(name='all')
@click.pass_context
def setup_all(ctx):
    """ Run all setup tasks.
    """
    ctx.forward(database)
    ctx.forward(homeserver)
    ctx.forward(email)
    ctx.forward(sso)
    ctx.forward(enable)

    ctx.forward(federation)
    ctx.forward(modules)
    ctx.forward(password_jwt)
    ctx.forward(push)
    ctx.forward(registration)
    ctx.forward(rooms)
    ctx.forward(threepid)
    ctx.forward(maintenance)

    confirm_signing_keys()

    from .restart import restart_all
    ctx.forward(restart_all)
    ctx.forward(webserver)
