# -*- coding: utf-8 -*-
# claradm
# Copyright (C) 2020-2023 Johannes Tiefenbacher
#
# claradm is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# claradm is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
import os
from pathlib import Path
import re
from setuptools import setup, find_packages
with open('README.md') as f:
    long_description = f.read()


def get_data_files():
    import distro
    from claradm.settings import (
        CLARION_BASE_HOMESERVER_DIR,
        CLARION_BASE_DIRECTORY_DIR,
        CLARION_BASE_WEBSERVER_DIR,
    )

    distro_id = distro.id()
    if distro_id == 'clearos':
        hardening = 'clarion_config/systemd/hardening/legacy'
    else:
        hardening = 'clarion_config/systemd/hardening/recent'

    systemd_dir = '/etc/systemd/system/'
    data_files = [
        (CLARION_BASE_DIRECTORY_DIR, [os.path.join('clarion_config/directory', 'sydent.conf')]),
        (CLARION_BASE_WEBSERVER_DIR, [os.path.join('clarion_config/nginx', 'clarion.conf')]),
        (f'{systemd_dir}/clarion-directory.service.d', [os.path.join(hardening, 'hardening.conf')]),
        (f'{systemd_dir}/clarion-server.service.d', [os.path.join(hardening, 'hardening.conf')]),
        (f'{systemd_dir}/clarion-sygnal.service.d', [os.path.join(hardening, 'hardening.conf')]),
        (f'{systemd_dir}/clarion-worker@.service.d', [os.path.join(hardening, 'hardening.conf')]),
    ]

    homeserver_cfg = Path('clarion_config/homeserver').glob('*.yaml')
    data_files.append((CLARION_BASE_HOMESERVER_DIR, [str(p) for p in homeserver_cfg]))

    logging_cfg = Path('clarion_config/logging').glob('*.yaml')
    data_files.append(('/etc/clarion/homeserver/logging', [str(p) for p in logging_cfg]))

    workers_cfg = Path('clarion_config/workers').glob('*.yaml')
    data_files.append(('/etc/clarion/homeserver/workers', [str(p) for p in workers_cfg]))

    service_files = [str(p) for p in Path('clarion_config/systemd').glob('*.service')]
    service_files.extend([str(p) for p in Path('clarion_config/systemd').glob('*.target')])
    data_files.append(('/lib/systemd/system', service_files))

    return data_files


def get_version():
    with open('claradm/__init__.py', 'r') as f:
        line = f.readline()
        match = re.match(r'__version__ = \'([\d\.]+)\'', line)

        if not match:
            raise ImportError("Can't read the version of claradm")

        version = match.group(1)
        return version


setup(
    name="claradm",
    version=get_version(),
    author="Dave Burkholder & Spencer Weaver & Johannes Tiefenbacher",
    author_email="dave@compassfoundation.io",
    description="Command line admin tool for Clarion (Based on Matrix Synapse homeserver)",
    long_description=long_description,
    long_description_content_type="text/markdown",
    url="https://code.compassfoundation.io/clarion/claradm",
    project_urls={
        "Documentation": "https://code.compassfoundation.io/clarion/claradm",
        "Source Code": "https://code.compassfoundation.io/clarion/claradm"
    },
    license="GPLv3+",
    classifiers=[
        "License :: OSI Approved :: GNU General Public License v3 or later (GPLv3+)",
        "Programming Language :: Python :: 3", "Programming Language :: Python :: 3.9",
        "Programming Language :: Python :: 3.10", "Programming Language :: Python :: 3.11",
        "Operating System :: OS Independent", "Topic :: System :: Systems Administration",
        "Topic :: Communications :: Chat", "Environment :: Console"
    ],
    zip_safe=True,
    packages=find_packages(exclude=["tests"]),
    data_files=get_data_files(),
    install_requires=[
        "Click>=7.1,<9.0",
        "requests",
        "lcrequests",
        "ruamel.yaml",
        "tabulate",
        "PyYaml",
        "click-option-group>=0.5.2",
        "dnspython",
    ],
    options={
        'bdist_rpm': {
            'pre_install': 'pre_install.sh',
            'post_install': 'post_install.sh',
        },
    },
    entry_points="""
        [console_scripts]
        claradm=claradm.cli:root
    """,
)
