import click
import json
import os
from pathlib import Path
import ruamel
from ruamel.yaml import YAML, RoundTripDumper
from ruamel.yaml.compat import StringIO
from secrets import token_urlsafe
from claradm.settings import (
    CLARION_BASE_HOMESERVER_DIR,
    CLARION_CLIENTS_DIR,
    CLARION_HOMESERVER_DIR,
    WEB_ROOT,
)
from typing import Iterable, Union
from .dict_utils import get_nested_key, set_nested_key
from .ini_cfg import BaseDirectoryConfig, DirectoryEmail, DirectoryGeneral
from .typehints import DerivedField

import sys

FILE_PATH = Union[str, Path]


class ClaradmYAML(YAML):
    """
    Convenience class for loading and dumping data.
    """
    # Some configurations will derive values from another
    # config class. List any dependencies here.
    CONFIG_DEPENDENCIES = ()
    CONFIG_FILE = ''
    FIELDS = ()
    SECRETS_FIELDS = ()

    # Fields with values that should be derived
    # from other values in the configuration.
    DERIVED_FIELDS: Iterable[DerivedField] = ()

    # When saving this class, also update Directory Server
    # config settings assigned by these classes
    DIRECTORY_SERVER_CLASSES: Iterable[BaseDirectoryConfig] = ()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._config = None

        # Before saving configuration, some classes may need to read other
        # config files to load entire context, so save compiled config.
        self._all_config_data = None

    def get(self, key, default=None):
        """
        Get value for specified key. If the key is colon-delimited,
        the dictionary is nested.

        data = {
            'email': {
                'smtp_host': 'mail.host.com',
                'smtp_port': 587,
            }
        }

        Key of "email:smtp_port" return 587, just as data['email']['smtp_port']
        """
        return get_nested_key(self.load_config(), key, default)

    def set(self, key, value):
        """
        Set key in config to the specified value. If key is colon-delimited,
        then dict is nested.

        data = {
            'email': {
                'smtp_host': 'mail.host.com',
                'smtp_port': 587,
            }
        }
        Key of "email:smtp_host" should set `value` as data['email']['smtp_port'] = 'mail.host.net'
        """
        cfg = self._config if self._config is not None else self.load_config()
        return set_nested_key(cfg, key, value)

    def load_config(self) -> dict:
        """
        Load config data from yaml file.
        """
        if self._config is not None:
            return self._config

        config_file = Path(f'{CLARION_HOMESERVER_DIR}/{self.CONFIG_FILE}')
        if not config_file.is_file():
            config_file = Path(f'{CLARION_BASE_HOMESERVER_DIR}/{self.CONFIG_FILE}')

            if not config_file.is_file():
                click.echo('{} does not exist'.format(config_file.name))
                return {}

        with open(config_file) as fp:
            local_cfg = self.load(fp.read()) or {}
            if local_cfg:
                self._config = local_cfg

        return self._config or {}

    def dump_data(self, data, stream=None, **kwargs):
        """
        Dump data to the specified stream.
        """
        inefficient = False
        if stream is None:
            inefficient = True
            stream = StringIO()

        yaml = YAML()
        yaml.dump(data, stream, **kwargs)

        if inefficient:
            return stream.getvalue()

    def save(self, file_path: FILE_PATH = '', **kwargs):
        with open(file_path or f'{CLARION_HOMESERVER_DIR}/{self.CONFIG_FILE}', 'w') as fp:
            self.dump_data(self.load_config(), fp, **kwargs)

    def collect_data(self):
        """
        Read data from config file and prompt user for updated values.
        """
        for field, prompt, default in self.FIELDS:
            current_value = self.get(field, default)
            value = click.prompt(prompt, default=current_value, show_default=True)
            if value != '':
                self.set(field, value)

        for field, prompt, default in self.SECRETS_FIELDS:
            current_value = self.get(field)
            value = click.prompt(
                prompt,
                default=current_value or 'skip',
                hide_input=True,
                show_default=False,
                prompt_suffix=' [press "enter" to leave unchanged]',
            )
            if value != '' and value != 'skip':
                self.set(field, value)
            elif not current_value:
                self.set(field, default() if callable(default) else default)

        self._all_config_data = dict(self.load_config().copy())

        if self.DERIVED_FIELDS:
            for ConfigClass in self.CONFIG_DEPENDENCIES:
                self._all_config_data.update(**dict(ConfigClass().load_config()))

            # If default values aren't set for derived fields, specify them here.
            print(self._all_config_data)
            for field in self.DERIVED_FIELDS:
                current_value = get_nested_key(self._all_config_data, field.name)
                if current_value and not field.overwrite:
                    continue

                self.set(field.name, field.value.format(**self._all_config_data))

        self.save()

        self.save_directory_cfg()

    def save_directory_cfg(self):
        """
        Save any Directory config variables that
        can be extracted from home server settings.
        """
        hs_cfg = self._all_config_data

        for ConfigClass in self.DIRECTORY_SERVER_CLASSES:
            cfg = ConfigClass()
            cfg.update_values(hs_cfg)


class HomeServerConfig(ClaradmYAML):
    """
    Handle homeserver.yaml configuration file.
    """
    CONFIG_FILE = 'homeserver.yaml'
    FIELDS = (
        ('entity_name', 'Company / Family Name', ''),
        ('subdomain', 'Server Subdomain', ''),
        ('clarion_service', 'Clarion Service Type', ''),
        ('report_stats', 'Report Stats', False),
    )
    SECRETS_FIELDS = (
        ('registration_shared_secret', 'Registration Shared Secret', token_urlsafe),
        ('macaroon_secret_key', 'Macaroon Secret Key', token_urlsafe),
        ('form_secret', 'Form Secret', token_urlsafe),
        ('worker_replication_secret', 'Worker Replication Secret', token_urlsafe),
    )
    DERIVED_FIELDS = (
        DerivedField('server_name', '{subdomain}.{clarion_service}', True),
        DerivedField('public_baseurl', 'https://{subdomain}.{clarion_service}', True),
        DerivedField(
            'signing_key_path', '/etc/clarion/homeserver/{subdomain}.{clarion_service}.signing.key',
            False
        ),
    )
    DIRECTORY_SERVER_CLASSES = [
        DirectoryGeneral,
    ]

    def collect_data(self):
        super().collect_data()
        self.save_well_known_client()
        self.save_well_known_server()
        self.save_client_config()

    def save_well_known_client(self):
        """
        Save well-known json files for client.
        """
        cfg_data = self._all_config_data
        server_name = f'{cfg_data["subdomain"]}.{cfg_data["clarion_service"]}'
        subdomain = cfg_data['subdomain']
        server = {
            "public_baseurl": f"https://{server_name}",
            "m.homeserver": {
                "base_url": f"https://{server_name}",
            },
            "m.identity_server": {
                "base_url": f"https://{subdomain}.clarion.directory",
            },
            "org.matrix.msc3814": "true",
        }
        with open(f'{WEB_ROOT}/well_known_client.json', 'w') as wkj:
            json.dump(server, wkj, indent=2)

    def save_well_known_server(self):
        """
        Save well-known json files for server.
        """
        cfg_data = self._all_config_data
        server = {"m.server": f'{cfg_data["subdomain"]}.{cfg_data["clarion_service"]}'}
        with open(f'{WEB_ROOT}/well_known_server.json', 'w') as wkj:
            json.dump(server, wkj, indent=2)

    def save_client_config(self):
        """
        Save element-web client config file.
        """
        cfg_data = self._all_config_data
        server_name = f'{cfg_data["subdomain"]}.{cfg_data["clarion_service"]}'
        subdomain = cfg_data['subdomain']
        client_data = {
            "default_server_config": {
                "m.homeserver": {
                    "base_url": f"https://{server_name}",
                    "server_name": server_name,
                },
                "m.identity_server": {
                    "base_url": f"https://{subdomain}.clarion.directory",
                },
            },
            "disable_custom_urls": True,
            "disable_guests": True,
            "disable_login_language_selector": False,
            "disable_3.im_login": False,
            "brand": "Clarion",
            "integrations_ui_url": None,
            "integrations_rest_url": None,
            "integrations_widgets_urls": None,
            "bug_report_endpoint_url": "#",
            "uisi_autorageshake_app": "element-auto-uisi",
            "default_country_code": "US",
            "show_labs_settings": False,
            "features": {},
            "default_federate": False,
            "default_theme": "light",
            "room_directory": {
                "servers": [],
            },
            "enable_presence_by_hs_url": {
                f"https://{server_name}": True,
            },
            "setting_defaults": {
                "breadcrumbs": True,
            },
            "m.identity_server": {
                "base_url": f"https://{subdomain}.clarion.directory",
            },
        }
        os.makedirs(CLARION_CLIENTS_DIR, exist_ok=True)
        with open(f'{CLARION_CLIENTS_DIR}/clarion_config.json', 'w') as wkj:
            json.dump(client_data, wkj, indent=2)


class EmailConfig(ClaradmYAML):
    """
    Handle email.yaml configuration file.
    """
    CONFIG_DEPENDENCIES = [
        HomeServerConfig,
    ]
    CONFIG_FILE = 'email.yaml'
    FIELDS = (
        ('email:smtp_host', 'SMTP Server', 'mail.daystar.io'),
        ('email:smtp_port', 'SMTP Port', 587),
        ('email:require_transport_security', 'Require Transport Security', True),
    )
    DERIVED_FIELDS = (
        DerivedField('email:app_name', 'Clarion {entity_name}', True),
        DerivedField('email:smtp_user', '{subdomain}@clarion.im', True),
        DerivedField(
            'email:notif_from', 'Clarion for {entity_name} <{subdomain}@clarion.im>', True
        ),
    )
    SECRETS_FIELDS = (('email:smtp_pass', 'SMTP password', token_urlsafe),)
    DIRECTORY_SERVER_CLASSES = [
        DirectoryEmail,
    ]


class FederationConfig(ClaradmYAML):
    """
    Handle federation.yaml configuration file.
    """
    CONFIG_FILE = 'federation.yaml'
    FIELDS = (
        ('allow_profile_lookup_over_federation', 'Allow Profile Lookup Over Federation', True),
        (
            'allow_device_name_lookup_over_federation', 'Allow Device Name Lookup Over Federation',
            True
        ),
    )


class MaintenanceConfig(ClaradmYAML):
    """
    Handle maintenance.yaml configuration file.
    """
    CONFIG_FILE = 'maintenance.yaml'
    FIELDS = (
        (
            'background_updates:background_update_duration_ms',
            'Background Update Duration in milliseconds', 500
        ),
        ('background_updates:sleep_enabled', 'Sleep Enabled', True),
        ('background_updates:sleep_duration_ms', 'Sleep Duration', 1000),
        ('background_updates:min_batch_size', 'Minimum Batch Size', 10),
        ('background_updates:default_batch_size', 'Default Batch Size', 100),
    )


class ModulesConfig(ClaradmYAML):
    """
    Handle modules.yaml configuration file.
    """
    CONFIG_FILE = 'modules.yaml'


class OidcConfig(ClaradmYAML):
    """
    Add Clavis to oidc.yaml configuration file.

    Any other SSO provider must be configured manually.
    """
    CONFIG_FILE = 'oidc.yaml'

    CONFIG_DEPENDENCIES = [
        HomeServerConfig,
    ]

    def collect_data(self):
        self._all_config_data = dict(self.load_config().copy())
        for ConfigClass in self.CONFIG_DEPENDENCIES:
            self._all_config_data.update(**dict(ConfigClass().load_config()))

        oidc_providers = self.get('oidc_providers', [])

        try:
            clavis = [op for op in oidc_providers if op['idp_id'] == 'clavis'][0]
        except IndexError:
            click.echo('Initial Clavis configuration not found')
            return

        self.set_client_whitelist()
        self.set_sso_attributes(clavis)
        self.set_attribute_requirements(clavis)

        for i, op in enumerate(oidc_providers):
            if op['idp_id'] == 'clavis':
                oidc_providers[i] = clavis
                break

        self.save_directory_cfg()
        self.save()

    def set_sso_attributes(self, data: dict):
        """
        Set Clavis client id and client secret.
        """
        current_value = data.get('client_id', '')
        data['client_id'] = click.prompt(
            'Clavis Client ID',
            default=current_value,
            show_default=True,
        )

        current_value = data.get('client_secret')
        value = click.prompt(
            'Clavis Client Secret',
            default=current_value or 'skip',
            hide_input=True,
            show_default=False,
            prompt_suffix=' [press "enter" to leave unchanged]',
        )
        if value and value != 'skip':
            data['client_secret'] = value
        return data

    def set_client_whitelist(self):
        """
        Ensure that client whitelist includes the web client URL.
        """
        cwl = self.get('sso:client_whitelist', [])
        try:
            domain = 'https://{subdomain}-web.{clarion_service}/'.format(**self._all_config_data)
        except KeyError:
            click.echo('Unable to set client whitelist')
            return

        if domain not in cwl:
            cwl.append(domain)
            self.set('sso:client_whitelist', cwl)

    def set_attribute_requirements(self, data: dict):
        """
        Set Group requirement to ClientID so that an SSO user is required to have
        been assigned to the Clarion server in Clavis to auto-register based on SSO name.
        """
        attr_requirements = data.get('attribute_requirements', [])
        group_name = data.get('client_id')
        if not group_name:
            return

        for requirement in attr_requirements:
            if requirement['value'] == group_name:
                return
        attr_requirements.append({'attribute': 'group', 'value': group_name})

        return data


class PasswordJwtConfig(ClaradmYAML):
    """
    Handle password_jwt.yaml configuration file.
    """
    CONFIG_FILE = 'password_jwt.yaml'
    FIELDS = (
        # Password Config
        ('password_config:enabled', 'Password Config Enabled', True),
        ('password_config:localdb_enabled', 'Local Database Enabled', True),
        ('password_config:policy:enabled', 'Password Config Policy Enabled', True),
        ('password_config:policy:minimum_length', 'Minimum Length', 15),
        ('password_config:policy:require_digit', 'Require Digit', True),
        ('password_config:policy:require_symbol', 'Require Symbol', True),
        ('password_config:policy:require_lowercase', 'Require Lowercase', True),
        ('password_config:policy:require_uppercase', 'Require Uppercase', True),

        # JWT Config
        ('jwt_config:enabled', 'JWT Enabled', True),
        ('jwt_config:algorithm', 'JWT Algorithm', 'HS256'),
    )
    SECRETS_FIELDS = (
        ('password_config:pepper', 'Password Pepper', token_urlsafe),
        ('jwt_config:pepper', 'JWT Shared Secret', token_urlsafe),
    )


class PushConfig(ClaradmYAML):
    """
    Handle push.yaml configuration file.
    """
    CONFIG_FILE = 'push.yaml'
    FIELDS = (
        ('push:include_content', 'Include message content in push notifications', False),
        ('push:group_unread_count_by_room', 'Group unread message count by room', True),
    )


class RegistrationConfig(ClaradmYAML):
    """
    Handle registration.yaml configuration file.
    """
    CONFIG_FILE = 'registration.yaml'
    FIELDS = (
        ('enable_registration', 'Enable account registration on this server', False),
        (
            'enable_registration_without_verification',
            'Enable account registration without verification', False
        ),
        ('session_lifetime', 'Session lifetime', '48h'),
        ('refreshable_access_token_lifetime', 'Refreshable access token lifetime', '30m'),
        ('refresh_token_lifetime', 'Refresh token lifetime', '48h'),
        ('nonrefreshable_access_token_lifetime', 'Non-refreshable access token lifetime', '48h'),
        ('allow_guest_access', 'Allow guest access', False),
        ('enable_set_displayname', 'Enable setting display name', True),
    )


class RoomsConfig(ClaradmYAML):
    """
    Handle rooms.yaml configuration file.
    """
    CONFIG_FILE = 'rooms.yaml'
    FIELDS = (
        ('user_directory:enabled', 'User Directory Enabled', True),
        ('user_directory:search_all_users', 'Search All Users', True),
        ('user_directory:prefer_local_users', 'Prefer Local Users', True),
        ('stats:enabled', 'Stats enabled', False),
        ('enable_room_list_search', 'Room List Search', False),
    )


class ThreePidConfig(ClaradmYAML):
    """
    Handle threepid.yaml configuration file.
    """
    CONFIG_FILE = 'threepid.yaml'
    FIELDS = (
        ('enable_3pid_lookup', 'Enable 3 PID Lookup', False),
        ('trusted_third_party_id_servers', 'Trusted Third Party Servers', []),
    )


__all__ = (
    'get_nested_key',
    'set_nested_key',
    'ClaradmYAML',
    'EmailConfig',
    'FederationConfig',
    'HomeServerConfig',
    'MaintenanceConfig',
    'ModulesConfig',
    'OidcConfig',
    'PasswordJwtConfig',
    'PushConfig',
    'RegistrationConfig',
    'RoomsConfig',
    'ThreePidConfig',
)
